/**
 * Copyright 2019-2021 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Text;

import com.nxp.swtools.common.ui.utils.editors.FileSelectionCellEditor;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;
import com.nxp.swtools.utils.text.TextBoxHelper;

/**
 * Class representing control of a file picker with modality to link files on disk
 * @author Tomas Rudolf - nxf31690
 */
public class FileLinkControl extends ScalarControl {

	/** Text with path to file */
	private @Nullable Text pathText;
	/** Button to select path to file */
	private @Nullable Button pathButton;

	/**
	 * @param child to create control for
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	public FileLinkControl(@NonNull ScalarConfig child, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(child, controlOptions, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ScalarControl#createMainControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @NonNull Control createMainControl(@NonNull Composite composite) {
		Composite group = new Composite(composite, getSwtStyle());
		group.setLayout(new GridLayout(2, false));
		Text text = new Text(group, SWT.BORDER);
		pathText = text;
		mainControlInternal = text;
		text.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));
		TextBoxHelper.attachModifyListeners(text, this::saveLink);
		Button button = new Button(group, SWT.PUSH);
		pathButton = button;
		button.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false));
		button.addSelectionListener(new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(@NonNull SelectionEvent e) {
				if (getChild().hasDirectoryType()) {
					String selectedPath = SWTFactoryProxy.INSTANCE.openDirectoryDialog(button.getShell(),
							Messages.get().FileControl_SelectDirectory, Messages.get().FileControl_SelectDirectory,
							UtilsText.safeString(getChild().getFilePath()));
					saveLink(selectedPath);
				} else {
					FileSelectionCellEditor.openFileDialog(button.getShell(), Messages.get().FileControl_SelectFile,
							UtilsText.safeString(getChild().getFilePath()), result -> saveLink(result));
				}
			}

		});
		createErrorDecoration(group, SWT.LEFT | SWT.TOP);
		return group;
	}

	/**
	 * Saves link to setting
	 * @param link to be saved or {@code null} when nothing changes
	 */
	void saveLink(@Nullable String link) {
		if (link == null) {
			return;
		}
		String newValue = getChild().prepareFileValue(link);
		if (!newValue.equals(getChild().getFilePath())) {
			controllerWrapper.getController().setValue(getChild(), newValue, FileLinkControl.class);
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarControl#updateMainContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateMainContent(@NonNull Control contentControl, UpdateType updateType) {
		if (updateType != UpdateType.PROBLEM_DECORATION) {
			Text pathTextLoc = pathText;
			Button pathButtonLoc = pathButton;
			if (pathTextLoc != null) {
				String text = UtilsText.safeString(getChild().getFilePath());
				Point previousSelection = pathTextLoc.getSelection();
				pathTextLoc.setText(text);
				pathTextLoc.setSelection(previousSelection.y);
			}
			if (pathButtonLoc != null) {
				if (getChild().hasDirectoryType()) {
					pathButtonLoc.setText(Messages.get().FileControl_SelectDirectory);
				} else {
					pathButtonLoc.setText(Messages.get().FileControl_SelectFile);
				}
			}
			contentControl.requestLayout();
		}
		if (mainControlInternal != null) {
			updateErrorDecoration(mainControlInternal);
		}
		updateErrorDecoration(contentControl);
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#setEnabledStateToMainControl(boolean)
	 */
	@Override
	protected void setEnabledStateToMainControl(boolean enabled) {
		super.setEnabledStateToMainControl(enabled);
		Text pathTextLoc = pathText;
		Button pathButtonLoc = pathButton;
		if (pathTextLoc != null) {
			pathTextLoc.setEnabled(enabled);
		}
		if (pathButtonLoc != null) {
			pathButtonLoc.setEnabled(enabled);
		}
	}
}
