/**
 * Copyright 2019-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.PartInitException;

import com.nxp.swtools.common.ui.utils.GcUtils;
import com.nxp.swtools.common.ui.utils.progress.UIJobHelper;
import com.nxp.swtools.common.ui.utils.swt.FontFactory;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.ScrolledCompositeHelper;
import com.nxp.swtools.common.ui.utils.views.GenericSelectionProvider;
import com.nxp.swtools.common.ui.utils.views.PartListener2Adapter;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.expression.ExpressionException;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.runtime.DelayedActionExecutorAdapter;
import com.nxp.swtools.common.utils.runtime.DelayedExecution;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.configuration.SwToolsProduct;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.controller.APeriphController;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.controller.events.EventTypes;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.controller.PeriphControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IComponentConfig;
import com.nxp.swtools.resourcetables.model.config.IConfigSetConfig;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.events.IEventListener;
import com.nxp.swtools.utils.events.ToolEvent;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.view.ToolView;
import com.nxp.swtools.validation.engine.IValidationProblem;
import com.nxp.swtools.validation.engine.ValidationEngineFactory;
import com.nxp.swtools.validation.engine.ValidationProblemListenerAdapter;

/**
 * Editor used for editing global component settings.
 * @author Tomas Rudolf - nxf31690
 */
public class GlobalComponentSettingView extends EditorViewBase {
	/** Delay before scrolling to config set of specified type after creation of view */
	private static final int DELAY_BEFORE_SCROLLING = 200;
	/** Logger of class */
	@SuppressWarnings("hiding") // Commonly used name that overrides the generic logger
	protected static final @NonNull Logger LOGGER = LogManager.getLogger(GlobalComponentSettingView.class);
	/** Number of columns in the component view title (Title, documentation link, enable button) */
	public static final int TITLE_ROW_COLS = 3;
	/** Number of columns in the component settings (name, peripheral, etc.) */
	public static final int COMPONENT_COLS = 2;
	/** Number of columns in the config set settings */
	public static final int CONFIG_SET_COLS = 3;
	/** ID of the editor */
	public static final @NonNull String ID = "com.nxp.swtools.periphs.gui.view.globalcomponentsettings"; //$NON-NLS-1$
	/** Minimal width of the editor in ems */
	private static final int MIN_WIDTH_EMS = 45;
	/** The "top-most" composite with scroll ability */
	protected ScrolledComposite scrolledComposite;
	/** Controller associated with the editor */
	protected final APeriphController controller = Controller.getInstance();
	/** {@link ComponentInstanceControl} associated with the editor */
	private @Nullable IChildControl componentSettingControl = null;
	/** Listeners which were registered and need to be unregistered during dispose */
	protected final @NonNull Collection<@NonNull IEventListener> registeredListeners = new ArrayList<>();
	/** Controls of all displayed config sets */
	protected final @NonNull Map<@NonNull IConfigSetConfig, ConfigSetControl> configSetControls = new HashMap<>();
	/** Resize listener instance */
	private @Nullable Listener resizeListener;
	/** view selection provider for other plugins */
	protected GenericSelectionProvider viewSelectionProvider = new GenericSelectionProvider();
	/** Key to retrieve testId from controls */
	private static final @NonNull String TEST_ID_KEY = SWTFactoryProxy.TEST_ID_KEY;

	/** Associated controller wrapper */
	protected final @NonNull IControllerWrapper controllerWrapper = PeriphControllerWrapper.getInstance();
	/** Action which needs to be performed after the view becomes visible */
	protected @NonNull ActionRequired requiredAction = ActionRequired.NO_ACTION;
	/** Listener for validations changes */
	private @NonNull ValidationProblemListenerAdapter validationsListener = new ValidationProblemListenerAdapter() {
		@Override
		public void validationProblemsChanged(@NonNull Collection<@NonNull IValidationProblem> problems) {
			if ((contentComposite != null) && !contentComposite.isDisposed()) {
				refreshIfVisible(UpdateType.PROBLEM_DECORATION);
			}
		}
	};

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.ViewPart#init(org.eclipse.ui.IViewSite)
	 */
	@Override
	public void init(IViewSite site) throws PartInitException {
		setSite(site);
		String secondaryId = site.getSecondaryId();
		if (secondaryId == null) {
			return;
		}
		site.setSelectionProvider(viewSelectionProvider);
		setPartName(Messages.get().GlobalComponentSettingView_GlobalSettings);
		setTitleImage(ToolView.getDecoratedImage(IToolsImages.IMAGE_EDITOR_COMPONENT_GLOBAL, -1));
		getSiteSafe().getPage().addPartListener(new IPartListener() {
			/* (non-Javadoc)
			 * @see org.eclipse.ui.IPartListener#partOpened(org.eclipse.ui.IWorkbenchPart)
			 */
			@Override
			public void partOpened(IWorkbenchPart part) {
				if (part.equals(GlobalComponentSettingView.this)) {
					DelayedExecution ex = new DelayedExecution(Messages.get().GlobalComponentSettingView_ScrollToConfigSet, new DelayedActionExecutorAdapter() {
						@Override
						public void runAction() {
							scrollTo(secondaryId, null);
						}
					}, DELAY_BEFORE_SCROLLING);
					ex.requestExecution();
				}
			}

			/* (non-Javadoc)
			 * @see org.eclipse.ui.IPartListener#partDeactivated(org.eclipse.ui.IWorkbenchPart)
			 */
			@Override
			public void partDeactivated(IWorkbenchPart part) {
				getSiteSafe().getPage().removePartListener(this);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.ui.IPartListener#partClosed(org.eclipse.ui.IWorkbenchPart)
			 */
			@Override
			public void partClosed(IWorkbenchPart part) {
				// Do nothing
			}

			/* (non-Javadoc)
			 * @see org.eclipse.ui.IPartListener#partBroughtToTop(org.eclipse.ui.IWorkbenchPart)
			 */
			@Override
			public void partBroughtToTop(IWorkbenchPart part) {
				// Do nothing
			}

			/* (non-Javadoc)
			 * @see org.eclipse.ui.IPartListener#partActivated(org.eclipse.ui.IWorkbenchPart)
			 */
			@Override
			public void partActivated(IWorkbenchPart part) {
				// Do nothing
			}
		});
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#dispose()
	 */
	@Override
	public void dispose() {
		ValidationEngineFactory.removeListener(validationsListener);
		for (IEventListener listener : registeredListeners) {
			controller.removeListener(listener);
		}
		super.dispose();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) {
		ScrolledComposite scrolledCompositeLoc = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.H_SCROLL);
		scrolledComposite = scrolledCompositeLoc;
		SWTFactoryProxy.INSTANCE.setTestId(scrolledCompositeLoc, TestIDs.PERIPHS_GLOBAL_COMPONENT_SETTING_VIEW_CONTENT);
		contentComposite = createDefaultComposite(scrolledCompositeLoc);
		GridLayoutComponents layout = new GridLayoutComponents(COMPONENT_COLS, false);
		layout.marginWidth = 8;
		layout.horizontalSpacing = 8;
		contentComposite.setLayout(layout);
		scrolledCompositeLoc.setContent(contentComposite);
		scrolledCompositeLoc.setExpandHorizontal(true);
		scrolledCompositeLoc.setExpandVertical(true);
		getSiteSafe().getPage().addPartListener(new PartListener2Adapter() {
			@Override
			public void partVisible(IWorkbenchPartReference partRef) {
				if (getViewSiteNonNull().getPart() == partRef.getPart(false)) {
					if (state == State.CREATED_HIDDEN) {
						// the view was hidden before, make it visible and refresh its content if required
						state = State.CREATED_VISIBLE;
						if (requiredAction == ActionRequired.RECREATE) {
							recreate();
						} else if (requiredAction == ActionRequired.REFRESH) {
							refreshSettings(UpdateType.NORMAL);
						}
						requiredAction = ActionRequired.NO_ACTION;
					} else if (state == State.READY_TO_CREATE) {
						// the view is ready to be created, create it
						state = State.CREATED_VISIBLE;
						recreate();
						requiredAction = ActionRequired.NO_ACTION;
					}
				}
			}
			@Override
			public void partHidden(IWorkbenchPartReference partRef) {
				if (getViewSiteNonNull().getPart() == partRef.getPart(false)) {
					if (state == State.CREATED_VISIBLE) {
						// the view was visible before, make it hidden and clear the required action flag
						state = State.CREATED_HIDDEN;
						requiredAction = ActionRequired.NO_ACTION;
					}
				}
			}
			@Override
			public void partClosed(IWorkbenchPartReference partRef) {
				if (getViewSiteNonNull().getPart() == partRef.getPart(false)) {
					getSiteSafe().getPage().removePartListener(this);
					state = State.UNINITIALIZED;
					requiredAction = ActionRequired.NO_ACTION;
				}
			}
		});
		// create content of the view asynchronously as it is possible that the view becomes hidden meanwhile, postpone creation in that case
		parent.getDisplay().asyncExec(() -> {
			if (state == State.UNINITIALIZED) {
				// the view is not created yet
				state = State.READY_TO_CREATE;
				IWorkbenchPage page = getSiteSafe().getPage();
				IWorkbenchPart part = getSiteSafe().getPart();
				if ((page == null) || (part == null)) {
					LOGGER.fine("[TOOL] GUI: page or part is set to null"); //$NON-NLS-1$
					return;
				}
				if (page.isPartVisible(part)) {
					// the view is still visible, create its content
					recreate();
					contentComposite.forceFocus();
					state = State.CREATED_VISIBLE;
				}
			}
		});
		registerEventListener();
		ValidationEngineFactory.addListener(validationsListener);
	}

	/**
	 * @return nonnull IViewSite guarded by assert
	 */
	protected @NonNull IViewSite getViewSiteNonNull() {
		IViewSite site = getViewSite();
		assert site != null;
		return site;
	}

	/**
	 * Register listener responsible for closing the editor if it is not needed anymore or refreshing the content.
	 */
	private void registerEventListener() {
		// register close listener
		IEventListener changesListener = new IEventListener() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.utils.events.IEventListener#handle(com.nxp.swtools.utils.events.ToolEvent)
			 */
			@Override
			public void handle(ToolEvent event) {
				// refresh the content asynchronously (avoid concurrent modification exception)
				if (event.originator != GlobalComponentSettingView.this) {
					if (state == State.CREATED_VISIBLE) {
						// the view is visible, recreate it
						Display display = Display.getCurrent();
						if (display != null) {
							display.asyncExec(() -> recreate());
						}
					} else if (state == State.CREATED_HIDDEN) {
						// the view is not visible, indicate that the view needs to be recreated
						requiredAction = ActionRequired.RECREATE;
					}
				}
			}
		};
		controller.addListener(EventTypes.CHANGE, changesListener);
		registeredListeners.add(changesListener);
		// register setting value change listener
		IEventListener settingListener = new IEventListener() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.utils.events.IEventListener#handle(com.nxp.swtools.utils.events.ToolEvent)
			 */
			@Override
			public void handle(ToolEvent event) {
				refreshIfVisible(UpdateType.NORMAL);
			}
		};
		controller.addListener(EventTypes.SETTING_CHANGE, settingListener);
		registeredListeners.add(settingListener);
		// register initialization change listener
		IEventListener initializationListener = new IEventListener() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.utils.events.IEventListener#handle(com.nxp.swtools.utils.events.ToolEvent)
			 */
			@Override
			public void handle(ToolEvent event) {
				refreshIfVisible(UpdateType.INITIALIZATION);
			}
		};
		controller.addListener(EventTypes.INITIALIZATION, initializationListener);
		registeredListeners.add(initializationListener);
	}

	/**
	 * Refresh the View in case it is visible (update title in case the View is not visible).
	 * @param updateType type of the update
	 */
	protected void refreshIfVisible(@NonNull UpdateType updateType) {
		if (state == State.CREATED_HIDDEN) {
			if (requiredAction != ActionRequired.RECREATE) {
				// the refresh should be scheduled only if the recreate is not scheduled yet
				requiredAction = ActionRequired.REFRESH;
			}
		} else {
			refreshSettings(updateType);
		}
	}

	/**
	 * Creates first row in the view.
	 * Contains Title, documentation link and enable/disable button if applicable.
	 * @param parent composite in which row should be created
	 */
	void createTitleRow(@NonNull Composite parent) {
		Composite titleRow = new Composite(parent, SWT.NONE);
		titleRow.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, COMPONENT_COLS, 1));
		// Fix color on enable button in dark theme
		titleRow.setBackgroundMode(SWT.INHERIT_FORCE);
		// set layout to TITLE_ROW_COLS columns
		final GridLayoutComponents titleRowLayout = new GridLayoutComponents(TITLE_ROW_COLS, false);
		titleRowLayout.marginWidth = titleRowLayout.marginHeight = 0;
		titleRow.setLayout(titleRowLayout);

		// create title
		Label title = new Label(titleRow, SWT.WRAP);
		FontFactory.changeStyle(title, SWT.BOLD);
		FontFactory.scaleFontSize(title, 1.5);

		// set title
		title.setText(Messages.get().GlobalComponentSettingView_GlobalSettings);
		title.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, false));
	}

	/**
	 * Create GUI elements for the selected component.
	 */
	void createContent() {
		Composite contentCompositeLoc = contentComposite;
		if (contentCompositeLoc != null) {
			createTitleRow(contentCompositeLoc);
			updateAllConfigSets(contentCompositeLoc);
		}
		String secondaryId = getViewSiteNonNull().getSecondaryId();
		if (secondaryId != null) {
			scrollTo(secondaryId, null);
		}
		scrolledComposite.setMinWidth(GcUtils.getEmWidth(scrolledComposite, true) * MIN_WIDTH_EMS);
		scrolledComposite.setMinHeight(contentComposite.computeSize(scrolledComposite.getClientArea().width, SWT.DEFAULT).y);
		resizeListener = new Listener() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
			 */
			@Override
			public void handleEvent(Event event) {
				if (!getSiteSafe().getPage().isPartVisible(GlobalComponentSettingView.this)) {
					return;
				}
				scrolledComposite.setMinHeight(contentComposite.computeSize(scrolledComposite.getClientArea().width, SWT.DEFAULT).y);
			}
		};
		contentComposite.addListener(SWT.Resize, resizeListener);
	}

	/**
	 * Updates all config set controls in given composite
	 * @param composite in which the new controls will be created
	 */
	void updateAllConfigSets(@NonNull Composite composite) {
		List<IConfigSetConfig> configSets = controller.getProfile().getConfiguredComponents().entrySet().stream()
				.map(x -> x.getValue().getGlobalConfigSet())
				.filter(Objects::nonNull)
				.collect(CollectorsUtils.toList());
		updateConfigSetsContent(configSets, composite);
	}

	/**
	 * Create controls for given config sets (or update if already exist).
	 * @param configSets for which to create GUI
	 * @param composite in which to create the GUI
	 */
	private void updateConfigSetsContent(@NonNull List<@NonNull IConfigSetConfig> configSets, @NonNull Composite composite) {
		// drop the unused controls
		configSetControls.keySet().retainAll(configSets);
		for (IConfigSetConfig configSet : configSets) {
			ConfigSetControl configSetControl = configSetControls.get(configSet);
			if ((configSetControl == null) || configSetControl.isDisposed()) { // control for the config set does not exist, create one
				configSetControl = (ConfigSetControl) ChildControlFactory.create(configSet, PeriphControllerWrapper.getInstance(), null);
				if (configSetControl == null) {
					LOGGER.log(Level.SEVERE, "[TOOL] Attempt to create control failed for config set: {0}", configSet); //$NON-NLS-1$
					continue;
				}
				// create GUI
				configSetControl.create(composite, CONFIG_SET_COLS);
			}
			configSetControl.update(UpdateType.NORMAL);
			configSetControls.put(configSet, configSetControl);
		}
	}

	/**
	 * Refresh content of the editor.
	 */
	void recreate() {
		if (contentComposite.isDisposed()) {
			return;
		}
		contentComposite.setRedraw(false);
		try {
			if (resizeListener != null) {
				contentComposite.removeListener(SWT.Resize, resizeListener);
			}
			if (componentSettingControl != null) {
				componentSettingControl.dispose();
			}
			for (ConfigSetControl configSetControl : configSetControls.values()) {
				configSetControl.dispose();
			}
			for (Control ctrl : contentComposite.getChildren()) {
				ctrl.dispose();
			}
			createContent();
		} catch (ExpressionException e) {
			e.log();
		} finally {
			contentComposite.setRedraw(true);
		}
		contentComposite.layout();
	}

	/**
	 * Open component editor.
	 * @param viewSite with binding to an editor
	 * @param componentType type of the component to edit
	 * @param activate whether to activate the view after creating
	 * @return {@code true} if an editor was successfully opened, {@code false} otherwise
	 */
	public static boolean open(@NonNull IViewSite viewSite, @NonNull String componentType, boolean activate) {
		IWorkbenchPage iWorkbenchPage = viewSite.getPage();
		if (iWorkbenchPage != null) {
			return open(iWorkbenchPage, componentType, null);
		}
		return false;
	}

	/**
	 * Open component editor.
	 * @param workbenchPage with binding to an editor
	 * @param componentType type of the component to edit
	 * @param problematicChild setting with problem to open the view on
	 * @return {@code true} if an editor was successfully opened, {@code false} otherwise
	 */
	public static boolean open(@Nullable IWorkbenchPage workbenchPage, @NonNull String componentType, @Nullable IChild problematicChild) {
		if (workbenchPage != null) {
			try {
				GlobalComponentSettingView openedView = (GlobalComponentSettingView) workbenchPage.showView(GlobalComponentSettingView.ID);
				workbenchPage.activate(openedView);
				openedView.scrollTo(componentType, problematicChild);
				return true;
			} catch (@SuppressWarnings("unused") PartInitException e) {
				return false;
			}
		}
		return false;
	}

	/**
	 * Scrolls view content to global config set of given component type
	 * @param componentType - type of component to scroll to
	 * @param problematicChild - setting with problem to open the view on
	 */
	void scrollTo(@NonNull String componentType, @Nullable IChild problematicChild) {
		Composite contentCompositeLoc = contentComposite;
		if (contentCompositeLoc != null) {
			Runnable runnable = () -> {
				IComponentConfig configuredComponent = controller.getConfiguredComponent(componentType);
				if (configuredComponent == null) {
					return;
				}
				IConfigSetConfig globalConfigSet = configuredComponent.getGlobalConfigSet();
				ConfigSetControl configSetControl = configSetControls.get(globalConfigSet);
				if (configSetControl != null) {
					String testId = UtilsText.EMPTY_STRING;
					Control mainControlLoc = configSetControl.mainControl;
					Control labelControlLoc = configSetControl.labelControl;
					if (labelControlLoc != null) {
						testId = UtilsText.safeString((String) labelControlLoc.getData(TEST_ID_KEY));
					}
					// Main composite has priority due to fact that in this case it scrolls to more content on screen than when using label
					if (mainControlLoc != null) {
						testId = UtilsText.safeString((String) mainControlLoc.getData(TEST_ID_KEY));
					}
					ScrolledCompositeHelper.scrollToControlWithId(contentCompositeLoc, testId);
					if (problematicChild != null) {
						if (SwToolsProduct.isUctProduct()) { // FIXME TomasR, Ionut T. v13 maintenance - include changes from UCT to new function, try new function in UCT, remove condition
							ComponentSettingViewHelper.selectChildrenOnPath(configSetControl, problematicChild.getId(), null, UpdateType.NORMAL);
						} else {
							ComponentSettingViewHelper.selectChildrenOnPath(configSetControl, problematicChild.getId(), UpdateType.NORMAL);
						}
						String childTestID = TestIDs.PERIPHS_SETTING_CONTROL + problematicChild.getId();
						Control control = ScrolledCompositeHelper.getControlWithTestId(contentCompositeLoc, childTestID);
						if (control != null) {
							control.setFocus();
						}
					}
				}
			};
			UIJobHelper.runUIJob(runnable, Messages.get().GlobalComponentSettingView_ScrollToConfigSet, false, contentCompositeLoc.getDisplay(), DELAY_BEFORE_SCROLLING);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.utils.view.ToolView#refreshStatus()
	 */
	@Override
	protected void refreshStatus() {
		refreshStatusBar();
	}

	/**
	 * Refresh settings, errors indicators, available modes.
	 * @param updateType type of the update
	 */
	void refreshSettings(@NonNull UpdateType updateType) {
		// update controls
		updateControls(updateType);
		if (updateType != UpdateType.PROBLEM_DECORATION) {
			// update dimensions of scrollable pane
			scrolledComposite.setMinHeight(contentComposite.computeSize(scrolledComposite.getClientArea().width, SWT.DEFAULT).y);
		}
	}

	/**
	 * Update all controls in config sets.
	 * @param updateType type of the update
	 */
	void updateControls(@NonNull UpdateType updateType) {
		contentComposite.setRedraw(false);
		try {
			if (componentSettingControl != null) {
				componentSettingControl.update(updateType);
			}
			configSetControls.values().forEach(x -> x.update(updateType));
		} finally {
			contentComposite.setRedraw(true);
		}
	}
}
