/**
 * Copyright 2019-2021 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.util.List;
import java.util.function.Function;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;

import com.nxp.swtools.common.ui.utils.swt.widgets.RadioButtonsGroup;
import com.nxp.swtools.common.ui.utils.swt.widgets.RadioButtonsGroupHorizontal;
import com.nxp.swtools.common.ui.utils.swt.widgets.RadioButtonsGroupVertical;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig.Type;
import com.nxp.swtools.resourcetables.model.data.setting.SetSetting.CustomItem;
import com.nxp.swtools.resourcetables.model.data.setting.SetSetting.Item;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

/**
 * Class representing control of a check-box setting configuration.
 * @author Tomas Rudolf - nxf31690
 */
public class ScalarRadioGroupControl extends ScalarControl {
	/** Id of value false */
	private static final @NonNull String ID_FALSE = "false"; //$NON-NLS-1$
	/** Id of value true */
	private static final @NonNull String ID_TRUE = "true"; //$NON-NLS-1$
	/** Label of value false */
	private static final @NonNull String LABEL_FALSE = "False"; //$NON-NLS-1$
	/** Label of value true */
	private static final @NonNull String LABEL_TRUE = "True"; //$NON-NLS-1$

	/**
	 * @param child to create control for
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	public ScalarRadioGroupControl(@NonNull ScalarConfig child, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(child, controlOptions, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ScalarControl#createMainControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @NonNull Control createMainControl(@NonNull Composite composite) {
		// FIXME TomasR v99 maintenance create separate classes for vertical and horizontal
		RadioButtonsGroup group;
		if (getControlOptions().isRadioGroupHorizontal()) {
			group = new RadioButtonsGroupHorizontal(composite, SWT.NONE);
			if (getControlOptions().isRadioGroupValueLabelHidden()) {
				group.setShowHorizontalHeader(false);
			}
			if (getChild().getType() == Type.BOOL) {
				Function<@NonNull Integer, @NonNull String> valueLabelSupplier = (col) -> (col.intValue() == 0) ? LABEL_TRUE : LABEL_FALSE;
				group.setHorizontalHeaderSupplier(valueLabelSupplier);
			} else if (getChild().getType() == Type.ENUM) {
				List<@NonNull Item> enumItems = getEnumItems();
				group.setHorizontalHeaderSupplier((col) -> UtilsText.safeString(enumItems.get(col.intValue()).getUINameString()));
			}
			if (getControlOptions().isRadioGroupIndicesHidden()) {
				group.setVerticalHeaderSupplier(null);
			} else {
				group.setVerticalHeaderSupplier((row) -> getChild().getUiName());
			}
		} else {
			// Default
			group = new RadioButtonsGroupVertical(composite, SWT.NONE);
			if (getControlOptions().isRadioGroupIndicesHidden()) {
				group.setHorizontalHeaderSupplier(null);
			} else {
				group.setHorizontalHeaderSupplier((col) -> getChild().getUiName());
			}
			if (getControlOptions().isRadioGroupValueLabelHidden()) {
				group.setVerticalHeaderSupplier(null);
			} else {
				if (getChild().getType() == Type.BOOL) {
					group.setVerticalHeaderSupplier((row) -> (row.intValue() == 0) ? LABEL_TRUE : LABEL_FALSE);
				} else if (getChild().getType() == Type.ENUM) {
					List<@NonNull Item> enumItems = getEnumItems();
					group.setVerticalHeaderSupplier((row) -> UtilsText.safeString(enumItems.get(row.intValue()).getUINameString()));
				}
			}
		}
		if (getChild().getType() == Type.BOOL) {
			group.setNumberOfButtons(2);
		} else if (getChild().getType() == Type.ENUM) {
			List<@NonNull Item> enumItems = getEnumItems();
			group.setNumberOfButtons(enumItems.size());
		}
		group.addListener(SWT.Selection, new Listener() {
			@Override
			public void handleEvent(@NonNull Event event) {
				int selectedButtonIndex = event.detail;
				String newVal = UtilsText.EMPTY_STRING;
				ScalarConfig scalarConfig = getChild();
				if (scalarConfig.getType() == Type.BOOL) {
					newVal = (selectedButtonIndex == 0) ? ID_TRUE : ID_FALSE;
				} else if (scalarConfig.getType() == Type.ENUM) {
					List<@NonNull Item> enumItems = getChild().getEnumItems();
					newVal = enumItems.get(selectedButtonIndex).getId();
				}
				changeModelValue(newVal);
			}
		});
		group.setData(RadioButtonsGroup.TOOLTIP_KEY, ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(getChild())));
		group.createHorizontalHeaders();
		group.createButtons();
		return group;
	}

	/**
	 * Get list of enum items to show
	 * @return list of enum items
	 */
	private @NonNull List<@NonNull Item> getEnumItems() {
		return getChild().getEnumItems().stream()
				.filter(x -> x.isAvailable(getChild().getExpressionContext()))
				.filter(x -> !x.getId().equals(CustomItem.CUSTOM_ID))
				.collect(CollectorsUtils.toList());
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarControl#updateMainContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateMainContent(@NonNull Control contentControl, UpdateType updateType) {
		if (updateType != UpdateType.PROBLEM_DECORATION) {
			if (mainControl instanceof RadioButtonsGroup) {
				RadioButtonsGroup group = (RadioButtonsGroup) mainControl;
				String value = child.getValue().toString();
				ScalarConfig scalarConfig = getChild();
				int selectedButton = 0;
				if (scalarConfig.getType() == Type.BOOL) {
					if (value.equals(ID_FALSE)) {
						selectedButton = 1;
					}
				} else if (scalarConfig.getType() == Type.ENUM) {
					List<@NonNull String> enumItemIds = getChild().getEnumItems().stream().map(item -> item.getId()).collect(CollectorsUtils.toList());
					int index = enumItemIds.indexOf(value);
					if (index != -1) {
						selectedButton = index;
					}
				}
				group.setData(RadioButtonsGroup.TOOLTIP_KEY, ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(getChild())));
				group.setEnabled(getChild().isEnabled());
				group.setCurrentSelection(selectedButton, true);
				group.updateLayout();
			}
		}
		updateErrorDecoration(contentControl);
	}
}
