/**
 * Copyright 2017-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.text.MessageFormat;
import java.util.Objects;
import java.util.logging.Level;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Text;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.expression.Expression;
import com.nxp.swtools.common.utils.expression.ExpressionException;
import com.nxp.swtools.common.utils.expression.IFunction;
import com.nxp.swtools.common.utils.expression.IValue;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig.Type;
import com.nxp.swtools.resourcetables.model.config.ScalarUtils;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.preferences.KEPreferences;
import com.nxp.swtools.utils.resources.ToolsColors.SwToolsColors;
import com.nxp.swtools.utils.text.TextBoxHelper;
import com.nxp.swtools.utils.text.TextBoxHelper.Status;

import animations.swt.ColorChanger;
import animations.swt.EaseSelector;

/**
 * Class representing control of a text setting configuration.
 * @author Juraj Ondruska
 */
public class ScalarTextControl extends ScalarControl {
	/** Maximal number of lines which is permitted */
	private static final int MAXIMAL_PERMITTED_LINES = 200;
	/** Time how long the animation will be performed (ms) */
	private static final int ANIMATION_TIME = 1500;
	
	/**
	 * @param child to create control for
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	public ScalarTextControl(@NonNull ScalarConfig child, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(child, controlOptions, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.ScalarControl#createMainControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @NonNull Control createMainControl(@NonNull Composite composite) {
		final Composite textComposite = new Composite(composite, SWT.NONE);
		GridLayoutComponents textCompositeLayout = new GridLayoutComponents();
		textCompositeLayout.marginWidth = textCompositeLayout.marginHeight = 0;
		textComposite.setLayout(textCompositeLayout);
		GridDataComponents textCompositeLayoutData = new GridDataComponents(SWT.FILL, SWT.CENTER, true, false);
		Text text = new Text(textComposite, getSwtStyle());
		text.setLayoutData(textCompositeLayoutData);
		SWTFactoryProxy.INSTANCE.setTestId(text, TestIDs.PERIPHS_SETTING_CONTROL + child.getId());
		mainControlInternal = text;
		TextBoxHelper.attachModifyListeners(text, (value) -> {
			String childValue = getChild().getStringValue();
			if (!Objects.equals(value, childValue)) {
				Expression stringValidationExpr = getChild().getStringValidationExpr();
				if (stringValidationExpr != null) {
					IValue resolvedExpression;
					try {
						resolvedExpression = stringValidationExpr.resolve(getChild().getExpressionContext());
					} catch (Exception e) {
						LOGGER.severe("[DATA] Expression resolving caused an error with message: " + e.getMessage()); //$NON-NLS-1$
						return;
					}
					IFunction functionReference;
					try {
						functionReference = resolvedExpression.getFunctionReference();
					} catch (ExpressionException e) {
						e.log();
						return;
					}
					Object result = functionReference.invokeOn(getChild().getExpressionContext(), value).getValue();
					if ((result instanceof Boolean) && !((Boolean) result).booleanValue()) {
						return;
					}
				}
				changeModelValue(value);
			}
		});
		TextBoxHelper.attachModifyErrorListener(() -> getChild().getStringValue(), text,
				value -> getStatus(value));
		createErrorDecoration(text, SWT.LEFT | SWT.TOP);
		addScrollListener(text, composite);
		return textComposite;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#getSwtStyle()
	 */
	@Override
	public int getSwtStyle() {
		int style = super.getSwtStyle();
		if (getControlOptions().getNumOfLines() != null) {
			style |= SWT.MULTI | SWT.V_SCROLL;
		}
		return style;
	}

	/**
	 * Get status based on value.
	 * @param value String value that is being checked
	 * @return {@link Status#INVALID} if the text can't be parsed to value according to setting's type
	 * {@link Status#VALUE_ERROR} if value of the text is invalid for the setting (e.g. out of range)
	 * {@link Status#OK} otherwise
	 */
	private @NonNull Status getStatus(@NonNull String value) {
		if (child.isEnabled()) {
			if (!getChild().isParseable(value)) {
				return Status.INVALID;
			}
			if ((getChild().getType() != Type.STRING) && !(ScalarUtils.isInRange(getChild(), value))) {
				return Status.VALUE_ERROR;
			}
			Expression stringValidationExpr = getChild().getStringValidationExpr();
			if (stringValidationExpr != null) {
				try {
					IFunction functionReference;
					try {
						functionReference = stringValidationExpr.resolve(getChild().getExpressionContext()).getFunctionReference();
					} catch (ExpressionException e) {
						String description = "[DATA] Lambda function is required in string valdation expression"; //$NON-NLS-1$
						LOGGER.severe(description);
						throw e;
					}
					Object result = functionReference.invokeOn(getChild().getExpressionContext(), value).getValue();
					if ((result instanceof Boolean) && !((Boolean) result).booleanValue()) {
						return Status.INVALID;
					}
				} catch (@SuppressWarnings("unused") IllegalArgumentException e) {
					return Status.INVALID;
				}
			}
		}
		return Status.OK;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarControl#updateMainContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateMainContent(@NonNull Control contentControl, UpdateType updateType) {
		if (updateType != UpdateType.PROBLEM_DECORATION) {
			Text textControl = (Text) mainControlInternal;
			if (textControl != null) {
				Object data = textControl.getLayoutData();
				if (data instanceof GridDataComponents) {
					GridDataComponents layoutData = (GridDataComponents) data;
					Integer multiLine = getControlOptions().getNumOfLines();
					if (multiLine != null) {
						int lines = multiLine.intValue();
						if (lines < 1) {
							lines = 1;
							LOGGER.log(Level.SEVERE, () -> MessageFormat.format("[DATA] UI_MULTI_COLUMN option was set to \"{0}\" in setting with ID: {1}. Using value 1 instead.", //$NON-NLS-1$
									multiLine, getChild().getId()));
						}
						if (lines > MAXIMAL_PERMITTED_LINES) {
							lines = MAXIMAL_PERMITTED_LINES;
							LOGGER.log(Level.WARNING, () -> MessageFormat.format("[DATA] Value of option UI_MULTI_COLUMN exceeds permitted limit of \"{0}\" lines in setting wit hid {1}", //$NON-NLS-1$
									Integer.valueOf(MAXIMAL_PERMITTED_LINES), getChild().getId()));
						}
						layoutData.heightHint = lines * textControl.getLineHeight();
					}
				}
				String stringValue = getChild().getStringValue();

				if (!Objects.equals(stringValue, textControl.getText())) {
					// update content only if it is changed
					int caretPos = textControl.getCaretPosition();
					textControl.setText(stringValue);
					textControl.setSelection(caretPos);
				}
				Color background = SwToolsColors.getColor(SwToolsColors.DEFAULT_LIST_BG);
				Color foreground = SwToolsColors.getColor(SwToolsColors.DEFAULT_FG);
				textControl.setData(TextBoxHelper.STATE_PROPERTY_KEY, TextBoxHelper.STATE_OK);
				if (getChild().getWarning() != null) {
					background = SwToolsColors.getColor(SwToolsColors.WARNING_FG);
					foreground = SwToolsColors.getColor(SwToolsColors.BLACK_FG);
					textControl.setData(TextBoxHelper.STATE_PROPERTY_KEY, TextBoxHelper.STATE_WARNING);
				}
				if (getChild().getError() != null) {
					background = SwToolsColors.getColor(SwToolsColors.ERROR_BG);
					foreground = SwToolsColors.getColor(SwToolsColors.ERROR_FG);
					textControl.setData(TextBoxHelper.STATE_PROPERTY_KEY, TextBoxHelper.STATE_ERROR);
				}
				
				if(KEPreferences.isAnimationsEnabled()) {
					ColorChanger cc = new ColorChanger(textControl, background, EaseSelector.SPIKE);
					cc.play(ANIMATION_TIME);
				} else {
					textControl.setBackground(background);
				}
				textControl.setForeground(foreground);
			}
			if (mainControlInternal != null) {
				updateErrorDecoration(mainControlInternal);
			}
			updateErrorDecoration(contentControl);
		}
	}
}
