/**
 * Copyright 2017-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import com.nxp.swtools.common.ui.utils.swt.FontFactory;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.resourcetables.model.config.StructConfig;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

/**
 * Class representing control of a struct setting configuration.
 * @author Juraj Ondruska
 * @author David Danaj (b57899/nxa30572)
 */
public class StructControl extends ChildProvidableControlBase {
	/** Label of the structure */
	protected @Nullable Control label;
	/** The Quick selection control */
	protected @Nullable QuickSelectionControl quickSelection;

	/**
	 * @param structConfig to create control for
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	public StructControl(@NonNull StructConfig structConfig, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(structConfig, controlOptions, controllerWrapper);
		if (getChild().getModelData().hasQuickSelections()) {
			quickSelection = new QuickSelectionControl(structConfig, controllerWrapper);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#create(org.eclipse.swt.widgets.Composite, int)
	 */
	@Override
	public void create(@NonNull Composite composite, int colSpan) {
		super.create(composite, colSpan);
		Control mainControlLoc = mainControl;
		// update layout data for control here
		if (mainControlLoc != null) {
			mainControlLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, colSpan, 1));
		}
		// FIXME TomasR v13 maintenance - this doesn't work as expected -> when struct is created in an array, colSpan in this method differs from "numColumns" in method "createControlStandard"
		// Therefore, setting layout data sets incorrect value here
		// consider refactor

		Control labelControlLoc = labelControl;
		if (labelControlLoc != null) {
			// label is inside of this struct -> label is a composite with text + quickSelection combo -> expand it to all columns of a struct
			labelControlLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, colSpan, 1));
			registerCopyPasteMenuOnLabel();
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#update(com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	public void update(@NonNull UpdateType updateType) {
		super.update(updateType);
		QuickSelectionControl quickSelectionControl = quickSelection;
		if (quickSelectionControl != null) {
			quickSelectionControl.update(updateType);
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.editor.AChildControl#createLabelControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @Nullable Control createLabelControl(@NonNull Composite composite) {
		QuickSelectionControl quickSelectionLoc = quickSelection;
		if ((quickSelectionLoc != null) && !quickSelectionLoc.isEmpty()) {
			Composite labelComposite = new Composite(composite, SWT.None);
			labelComposite.setLayout(new GridLayoutComponents(2, false));
			Control labelLoc = super.createLabelControl(labelComposite);
			if (labelLoc != null) {
				FontFactory.changeStyle(labelLoc, SWT.BOLD);
				Control actualLabel = labelControlInternal;
				if (actualLabel != null) {
					createErrorDecoration(actualLabel, SWT.LEFT | SWT.TOP);
				}
				labelLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.BOTTOM, true, false));
				label = labelLoc;
			}
			quickSelectionLoc.create(labelComposite, labelLoc == null ? 2 : 1);
			return labelComposite;
		} else {
			// the quick selection won't be displayed, do not create unnecessary composite
			Control labelLoc = super.createLabelControl(composite);
			if (labelLoc != null) {
				FontFactory.changeStyle(labelLoc, SWT.BOLD);
				Control actualLabel = labelControlInternal;
				if (actualLabel != null) {
					createErrorDecoration(actualLabel, SWT.LEFT | SWT.TOP);
				}
				label = labelLoc;
			}
			return labelLoc;
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#updateLabelContent(org.eclipse.swt.widgets.Control, com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateLabelContent(@NonNull Control labelCtrl, @NonNull UpdateType updateType) {
		Control labelLoc = label;
		if (labelLoc != null) {
			controlOptions.labelBold(true);
			super.updateLabelContent(labelLoc, updateType);
			String toolTipText = ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(child));
			SWTFactoryProxy.INSTANCE.setHtmlTooltip(labelLoc, toolTipText);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#fillTopOfMainControl(org.eclipse.swt.widgets.Composite, int)
	 */
	@Override
	protected void fillTopOfMainControl(@NonNull Composite composite, int numColumns) {
		QuickSelectionControl quickSelectionLoc = quickSelection;
		if (controlOptions.isLabelHidden() && (quickSelectionLoc != null) && !quickSelectionLoc.isEmpty()) { // if the label is hidden, create the quick selection inside of the main control
			quickSelectionLoc.create(composite, numColumns);
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#dispose()
	 */
	@Override
	public void dispose() {
		if (quickSelection != null) {
			quickSelection.dispose();
		}
		if (label != null) {
			label.dispose();
			label = null;
		}
		super.dispose();
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#getChild()
	 */
	@Override
	public @NonNull StructConfig getChild() {
		return (StructConfig) super.getChild();
	}
}
