/**
* Copyright 2018-2021 NXP
* Created: Feb 14, 2018
*/
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import com.nxp.swtools.common.ui.utils.GcUtils;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig;

/**
 * Common base for array settings interconnected in the GUI and sharing controls (e.g. row of a table).
 * @author Viktar Paklonski
 */
public abstract class AArrayControlGroup extends AArrayControlInternal {
	/** menu control of a child control */
	protected @Nullable ArrayControlItemMenuContext menuControl;
	/** calculated column width unit size */
	private @Nullable Integer emWidth;

	/**
	 * Constructor.
	 * @param arrayConfig for which to create the GUI
	 * @param controlOptions for this child
	 * @param controllerWrapper containing the generic controller
	 */
	protected AArrayControlGroup(@NonNull ArrayConfig arrayConfig, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(arrayConfig, controlOptions, controllerWrapper);
	}

	/**
	 * Computes unit size (in points) of the column width
	 * @param control table composite
	 * @return unit size in points on success, {@code 0} otherwise
	 */
	protected int computeColumnWidthUnitSize(@NonNull Composite control) {
		Integer emWidthLoc = emWidth;
		if (emWidthLoc == null) {
			emWidthLoc = Integer.valueOf(GcUtils.getEmWidth(control, true));
			emWidth = emWidthLoc;
		}
		return emWidthLoc.intValue();
	}

	/**
	 * Associates child setting controls with the specified menu control.
	 * @param menu control instance
	 */
	protected void registerControl(@NonNull ArrayControlItemMenuContext menu) {
		removeMenuControl();
		menuControl = menu;
	}

	/**
	 * Destroys a previously associated setting menu control.
	 */
	protected void removeMenuControl() {
		if (menuControl != null) {
			menuControl.dispose();
			menuControl = null;
		}
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildControlBase#createLabelControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public @Nullable Control createLabelControl(@NonNull Composite composite) {
		return null;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ChildProvidableControlBase#dispose()
	 */
	@Override
	public void dispose() {
		removeMenuControl();
		super.dispose();
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#updateButtons(com.nxp.swtools.periphs.gui.view.
	 * componentsettings.IChildControl.UpdateType)
	 */
	@Override
	protected void updateButtons(@Nullable UpdateType updateType) {
		if (!areButtonsEnabled()) {
			updateType = UpdateType.FORCE_DISABLE;
		}
		if (menuControl != null) {
			menuControl.update(updateType);
		}
		updateAddButton(updateType);
		updateRemoveButton(updateType);
		updateUpButton(updateType);
		updateDownButton(updateType);
	}
}
