/**
 * Copyright 2017-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.configuration.properties.SWToolsProperties;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.componentsettings.ControlOptions;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl;
import com.nxp.swtools.periphs.model.data.XMLConstants;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IRoot;
import com.nxp.swtools.resourcetables.model.config.ISettingConfig;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig.Type;
import com.nxp.swtools.resourcetables.model.data.SettingOptions;
import com.nxp.swtools.resourcetables.model.data.setting.IBaseModel;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;

/**
 * Class representing tabular representation of an array config with items represented as rows.
 * @author Juraj Ondruska
 * @author Marek Ciz
 * @author Viktar Paklonski
 */
public class ArrayControlTabularVertical extends AArrayControlTabular {

	/** Offset in column indexing without indices in table */
	private static final int OFFSET_WITHOUT_INDICES = 2;
	/** Offset in column indexing with indices in table */
	private static final int OFFSET_WITH_INDICES = 3;
	/** List of indexes which are permanently hidden */
	private List<Integer> permanentlyHiddenColumns = new ArrayList<>();

	/**
	 * Constructor.
	 * @param arrayConfig for which to create the table
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	protected ArrayControlTabularVertical(@NonNull ArrayConfig arrayConfig, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(arrayConfig, controlOptions, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#tableViewerHandling(org.eclipse.jface.viewers.TableViewer)
	 */
	@Override
	protected void tableViewerHandling(@NonNull TableViewer tableViewerLoc) {
		final Table table = tableViewerLoc.getTable();
		tableViewerLoc.addSelectionChangedListener(new ISelectionChangedListener() {
			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
			 */
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				ISelection selection = event.getSelection();
				if (selection instanceof StructuredSelection) {
					Object selectedElement = ((StructuredSelection) selection).getFirstElement();
					if (selectedElement instanceof IChildControl) {
						IChildControl newSelection = (IChildControl) selectedElement;
						setSelectedChild(newSelection);
						updateButtons(getControlUpdateType());
					}
				}
			}
		});
		createDecoratorTableViewerColumn(tableViewerLoc);
		// create the indices column only in case indices are allowed
		if (shouldShowIndices()) {
			createIndicesTableViewerColumn(tableViewerLoc);
		}
		final IRoot root = arrayConfig.getChildContext().getRoot();
		// create some temporary fake settingConfig to retrieve proper column titles
		// FIXME TomasR v13 maintenance - Analyze usage of "0" as id and name. Creates problem with quick selection. ".somethingInStructure" is not valid id
		ISettingConfig childForHeaderLabels = root.getConfigFactory().createSettingConfig(UtilsText.EMPTY_STRING, UtilsText.EMPTY_STRING,
				arrayConfig.getModelData().getReferenceType(), arrayConfig.getChildContext(), root.getMcu());
		childForHeaderLabels.delayedInitAfterCreation();
		// create tableViewers columns based on the labels retrieved from the fake settingConfig
		for (@NonNull IChild childLoc : getSettingsFlat(childForHeaderLabels)) {
			IBaseModel baseModel = childLoc.getModelData();
			if (baseModel != null) {
				createTableViewerColumnVertical(baseModel, tableViewerLoc, shouldShowIndices());
			}
		}
		// remove the temporary fake settingConfig
		arrayConfig.remove(childForHeaderLabels);
		setTableLayout(table, arrayConfig.getId(), true);
		if (SWToolsProperties.isVerificationOn()) {
			if (arrayConfig.getModelData().getUINameExprString() != null) {
				LOGGER.log(Level.WARNING, String.format("[DATA] %1s is used as table title from %2s; table titles are only updated on component (re-)open", //$NON-NLS-1$
						XMLConstants.LABEL_EXPR, arrayConfig));
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#getMenuContext(org.eclipse.jface.viewers.TableViewer)
	 */
	@Override
	protected @NonNull ArrayControlItemMenuContext getMenuContext(@NonNull TableViewer tableViewerLoc) {
		return new ArrayControlItemMenuContextVertical(this, tableViewerLoc);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlGroup#createLabelControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public Control createLabelControl(Composite composite) {
		return null;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#updateTableViewer(org.eclipse.jface.viewers.TableViewer)
	 */
	@Override
	protected void updateTableViewer(@NonNull TableViewer viewerLoc) {
		// input updated data
		viewerLoc.setInput(children);
		if (contentContainer != null) {
			contentContainer.requestLayout();
		}
		IChildControl childToSelect = getChildToSelect();
		if (childToSelect != null) {
			viewerLoc.setSelection(new StructuredSelection(childToSelect), true);
		}
	}

	/**
	 * Create table column from given configuration in given Vertical table
	 * @param baseModel of the newly created column
	 * @param tableViewerLoc in which the new column will be created
	 * @param isIndices if indices column is required
	 */
	protected void createTableViewerColumnVertical(@NonNull IBaseModel baseModel, @NonNull TableViewer tableViewerLoc, boolean isIndices) {
		// decide if indices column is generated
		final int rightOffset = isIndices ? OFFSET_WITH_INDICES : OFFSET_WITHOUT_INDICES;
		final int columnIndex = (tableViewerLoc.getTable().getColumnCount() + 1 + permanentlyHiddenColumns.size()) - rightOffset;
		if (baseModel.isOptionAvailable(SettingOptions.UI_NOT_VISIBLE_PERMANENT, getChild().getExpressionContext())) {
			permanentlyHiddenColumns.add(Integer.valueOf(columnIndex));
			return;
		}
		final TableViewerColumn viewerColumn = new TableViewerColumn(tableViewer, SWT.NONE);
		viewerColumn.getColumn().addControlListener(new ControlAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.ControlAdapter#controlResized(org.eclipse.swt.events.ControlEvent)
			 */
			@Override
			public void controlResized(ControlEvent e) {
				updateScrollSize();
			}
		});
		viewerColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				String shownText = null;
				IChild currentChild = getChild(element, columnIndex);
				// assign child for column listener
				if ((currentChild instanceof ScalarConfig)) {
					if (currentChild.isAvailable()) {
						ScalarConfig currentSettingLoc = (ScalarConfig) currentChild;
						shownText = AArrayControlTabular.getText(currentSettingLoc);
					} else {
						shownText = NOT_APPLICABLE;
					}
				}
				return shownText;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getFont(java.lang.Object)
			 */
			@Override
			public Font getFont(Object element) {
				IChild currentChild = getChild(element, columnIndex);
				if ((currentChild != null) && (!currentChild.isAvailable())) {
					return italicFont;
				}
				return null;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getImage(java.lang.Object)
			 */
			@Override
		    public Image getImage(Object element) {
				IChild currentChild = getChild(element, columnIndex);
				if (currentChild != null) {
					Image image = getImageOfLinkedSettingState(currentChild);
					if (image != null) {
						return image;
					}
					return AArrayControlTabular.getImage(currentChild, getControlUpdateType());
				}
				return null;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				IChild currentChild = getChild(element, columnIndex);
				if (currentChild != null) {
					if (!currentChild.isEnabled()) {
						return DISABLED_TEXT_COLOR;
					}
					if (currentChild.isOptionSet(SettingOptions.UI_SETTING_LINK)) {
						if (!currentChild.isOptionAvailable(SettingOptions.UI_SETTING_LINK)) {
							return DISABLED_TEXT_COLOR;
						}
						return LINK_COLOR;
					}
				}
				return super.getForeground(element);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getBackground(java.lang.Object)
			 */
			@Override
			public Color getBackground(Object element) {
				IChild currentChild = getChild(element, columnIndex);
				if (currentChild != null) {
					// decorate background with proper color
					return AArrayControlTabular.getBackgroundColor(currentChild);
				}
				return super.getBackground(element);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.CellLabelProvider#getToolTipText(java.lang.Object)
			 */
			@Override
			public String getToolTipText(Object element) {
				String tooltip = UtilsText.EMPTY_STRING;
				IChild currentChild = getChild(element, columnIndex);
				if (currentChild != null) {
					if (currentChild.isAvailable()) {
						tooltip = getToolTip(currentChild);
					} else {
						tooltip = NOT_APPLICABLE;
					}
				}
				return tooltip;
			}
		});
		addEditingSupport(tableViewerLoc, viewerColumn, columnIndex);
		// set title to a column
		final TableColumn column = viewerColumn.getColumn();
		String title = baseModel.getUIName(arrayConfig.getExpressionContext());
		column.setText(title);
		column.setToolTipText(title);
		column.pack();
		setColumnWidth(column, columnIndex + (getControlOptions().isArrayIndicesHidden() ? 0 : 1));
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#editingSupportGetChildControl(java.lang.Object, int)
	 */
	@Override
	protected @Nullable IChildControl editingSupportGetChildControl(@NonNull Object element, int index) {
		if (element instanceof IChildControl) {
			return (IChildControl) element;
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#editingSupportSetValue(org.eclipse.jface.viewers.TableViewer, java.lang.Object, int, java.lang.Object)
	 */
	@Override
	protected void editingSupportSetValue(@NonNull TableViewer tableViewer, @NonNull Object element, int index, @Nullable Object userInputValue) {
		IChild currentChild = getChild(element, index);
		if (currentChild instanceof ScalarConfig) {
			ScalarConfig currentSettingLoc = ((ScalarConfig) currentChild);
			// check-boxes options
			if (currentSettingLoc.getType() == Type.ENUM) {
				// Handled by listener of combo box in {@link #getCellEditor}
			} else if ((currentSettingLoc.getType() == Type.INTEGER) || (currentSettingLoc.getType() == Type.FLOAT)
					|| (currentSettingLoc.getType() == Type.STRING)) {
				controllerWrapper.getController().setValue(currentSettingLoc,
						(userInputValue == null) ? UtilsText.EMPTY_STRING : UtilsText.safeString(userInputValue.toString()), tableViewer);
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#editingSupportGetValue(java.lang.Object, int)
	 */
	@Override
	protected @Nullable Object editingSupportGetValue(@NonNull Object element, int index) {
		String selectedItem = UtilsText.EMPTY_STRING;
		IChild currentChild = getChild(element, index);
		if ((currentChild instanceof ScalarConfig) && currentChild.isAvailable()) {
			selectedItem = AArrayControlTabular.getText((ScalarConfig) currentChild);
		}
		return selectedItem;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#editingSupportCanEdit(java.lang.Object, int)
	 */
	@Override
	protected boolean editingSupportCanEdit(@NonNull Object element, int index) {
		boolean canEdit = true;
		if (element instanceof IChildControl){
			List<@NonNull IChild> childList = getSettingsFlat(((IChildControl) element).getChild());
			IChild currentChild = childList.get(index);
			canEdit = isCellEnabled(currentChild);
		}
		return canEdit;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#editingSupportSelectChild(java.lang.Object, int)
	 */
	@Override
	protected IChild editingSupportSelectChild(@NonNull Object element, int index) {
		return getChild(element, index);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlTabular#editingSupportPostAction(org.eclipse.jface.viewers.TableViewer, org.eclipse.jface.viewers.TableViewerColumn)
	 */
	@Override
	protected void editingSupportPostAction(@NonNull TableViewer viewer, @NonNull TableViewerColumn viewerColumn) {
		// Intentionally empty
	}

	/**
	 * Get proper child from the element.
	 * @param element from the table's input
	 * @param column of the child
	 * @return child or {@code null} if there is no child for the current column in the element
	 */
	protected @Nullable IChild getChild(@Nullable Object element, int column) {
		if (element instanceof IChildControl){
			List<@NonNull IChild> childList = getSettingsFlat(((IChildControl) element).getChild());
			if ((column >= 0) && (column < childList.size())) {
				return childList.get(column);
			}
		}
		return null;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlInternal#getSelection()
	 */
	@Override
	public ISettingConfig getSelection() {
		TableViewer tableViewerLoc = tableViewer;
		if (tableViewerLoc != null) {
			ISelection selection = tableViewerLoc.getSelection();
			return super.getChildFromSelection(selection);
		}
		return null;
	}

	/**
	 * Creates column of Indices numbers
	 * @param viewer table viewer of the table
	 */
	protected void createIndicesTableViewerColumn(@NonNull TableViewer viewer) {
		// first column in the table with left align
		final TableViewerColumn indicesViewerColumn = new TableViewerColumn(viewer, SWT.LEFT, 1);
		final TableColumn indicesColumn = indicesViewerColumn.getColumn();
		indicesColumn.setWidth(INDICES_COLUMN_MIN_WIDTH);
		// First set default value, then try to adjust the width by size specified in data
		setColumnWidth(indicesColumn, 0);
		indicesColumn.setAlignment(SWT.LEFT);
		indicesColumn.setText("#"); //$NON-NLS-1$
		indicesViewerColumn.setLabelProvider(new ColumnLabelProvider() {
			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public String getText(Object element) {
				String indices = UtilsText.EMPTY_STRING;
				if (element instanceof IChildControl) {
					indices = ((IChildControl) element).getChild().getUiName();
				}
				return indices;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getForeground(java.lang.Object)
			 */
			@Override
			public Color getForeground(Object element) {
				if (element instanceof IChildControl) {
					if (!((IChildControl) element).getChild().isEnabled()) {
						return DISABLED_TEXT_COLOR;
					}
				}
				return super.getForeground(element);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getBackground(java.lang.Object)
			 */
			@Override
			public Color getBackground(Object element) {
				if (element instanceof IChildControl) {
					return AArrayControlTabular.getBackgroundColor(((IChildControl) element).getChild());
				}
				return super.getBackground(element);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getToolTipText(java.lang.Object)
			 */
			@Override
			public String getToolTipText(Object element) {
				String tooltip = UtilsText.EMPTY_STRING;
				if (element instanceof IChildControl) {
					tooltip = ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(((IChildControl) element).getChild()));
				}
				return tooltip;
			}
		});
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.internal.AArrayControlGroup#dispose()
	 */
	@Override
	public void dispose() {
		super.dispose();
		permanentlyHiddenColumns.clear();
	}
}
