/**
 * Copyright 2019-2020 NXP
 */
package com.nxp.swtools.periphs.gui.view.provider;

import java.text.MessageFormat;
import java.util.List;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.controller.PeriphControllerWrapper;
import com.nxp.swtools.provider.configuration.dependencies.IDependency;
import com.nxp.swtools.provider.configuration.storage.StorageDependencyFeature;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.validation.engine.IBaseProblem;
import com.nxp.swtools.validation.engine.IValidationProblem;

/**
 * Problem handler used for responding to peripheral uninitialized problems.
 * @author Juraj Ondruska
 */
public final class PeripheralEnableResolveHandler implements IPeripheralsProblemResolveHandler {
	/** Singleton instance of the class */
	public static final @NonNull PeripheralEnableResolveHandler INSTANCE = new PeripheralEnableResolveHandler();

	/** Associated controller wrapper*/
	protected final @NonNull IControllerWrapper controllerWrapper = PeriphControllerWrapper.getInstance();
	/**
	 * Private constructor avoids creating instances from outside.
	 */
	private PeripheralEnableResolveHandler() {
		// intentionally empty
	}

	/**
	 * Returns list of component configuration instances that are located on peripheral which is subject of given problem
	 * @param problem to get peripheral name
	 * @return list of component configuration instances
	 */
	public @NonNull List<@NonNull IComponentInstanceConfig> getConfigInstancesOfProblem(@NonNull IValidationProblem problem) {
		String peripheralInstance = problem.getDependency().getResourceId();
		List<@NonNull IComponentInstanceConfig> componentInstanceConfigs = controllerWrapper.getController().getComponentInstanceConfigs(peripheralInstance);
		return componentInstanceConfigs;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.provider.IProblemResolveHandlerExt#canHandle(com.nxp.swtools.validation.engine.IBaseProblem)
	 */
	@Override
	public boolean canHandle(@NonNull IBaseProblem problem) {
		if (problem instanceof IValidationProblem) {
			IValidationProblem validationProblem = (IValidationProblem) problem;
			if (IDependency.RESOURCE_PERIPHERAL.equals(validationProblem.getDependency().getResourceType())) {
				boolean isInit = StorageDependencyFeature.FEATURE_INITIALIZED.equals(validationProblem.getFeature().getName());
				boolean areIntancesOnPeripheral = !getConfigInstancesOfProblem(validationProblem).isEmpty();
				return (isInit && areIntancesOnPeripheral);
			}
		}
		return false;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.validation.engine.IProblemResolveHandler#resolveProblem(com.nxp.swtools.validation.engine.IBaseProblem)
	 */
	@Override
	public boolean resolveProblem(@NonNull IBaseProblem problem) {
		if (problem instanceof IValidationProblem) {
			IValidationProblem validationProblem = (IValidationProblem) problem;
			controllerWrapper.getController().setComponentInstancesEnabled(getConfigInstancesOfProblem(validationProblem), true, getClass());
			return true;
		}
		return false;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.validation.engine.IProblemResolveHandler#getResolveProblemLabel(com.nxp.swtools.validation.engine.IBaseProblem)
	 */
	@Override
	public @NonNull String getResolveProblemLabel(@NonNull IBaseProblem problem) {
		if (problem instanceof IValidationProblem) {
			IValidationProblem validationProblem = (IValidationProblem) problem;
			List<@NonNull String> instanceNames = getConfigInstancesOfProblem(validationProblem).stream()
					.map(x -> x.getName())
					.collect(CollectorsUtils.toList());
			return MessageFormat.format(Messages.get().PeripheralEnableResolveHandler_Enable, instanceNames);
		}
		return UtilsText.EMPTY_STRING;
	}
}
