/**
 * Copyright 2018-2020 NXP
 */
package com.nxp.s32ds.rcp.modular.installer.ui.internal;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.equinox.p2.ui.AcceptLicensesWizardPage;
import org.eclipse.jface.dialogs.ControlEnableState;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.dialogs.IPageChangeProvider;
import org.eclipse.jface.dialogs.IPageChangedListener;
import org.eclipse.jface.dialogs.IPageChangingListener;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.PageChangedEvent;
import org.eclipse.jface.dialogs.PageChangingEvent;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.operation.ModalContext;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.util.Policy;
import org.eclipse.jface.util.SafeRunnable;
import org.eclipse.jface.wizard.IWizard;
import org.eclipse.jface.wizard.IWizardContainer2;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.ProgressMonitorPart;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Layout;
import org.eclipse.swt.widgets.Shell;
import org.osgi.service.prefs.Preferences;

import com.nxp.s32ds.rcp.modular.installer.ui.MIMessages;
import com.nxp.s32ds.rcp.modular.installer.ui.MIRegistryImages;
import com.nxp.s32ds.rcp.modular.installer.ui.window.MIGridLayoutFactory;
import com.nxp.s32ds.rcp.modular.installer.ui.window.MIWizardUtil;
import com.nxp.s32ds.rcp.modular.installer.ui.window.pages.MIWizardPackPage;
import com.nxp.s32ds.rcp.modular.installer.ui.window.pages.MIWizardSumPage;
import com.nxp.s32ds.rcp.modular.internal.core.preferences.ModularPreferenceConstants;

public class MIWindowDialog extends TitleAreaDialog implements IWizardContainer2, IPageChangeProvider {

	private static final char CHAR_ARROW_RIGHT = '>';
	private static final char CHAR_ARROW_LEFT = '<';
	private static final int LAST_JOB_FINISH_FLAG = -1;
	private static final int DEF_MIN_HEIGHT = 225;
	private static final int DEF_MIN_WIDTH = 300;
	private static final String FOCUS_CONTROL = "focusControl"; //$NON-NLS-1$
	private static final int RESTORE_DELAY = 500;

	private static final int BUTTON_WIDTH_190 = 190;
	private static final int BUTTON_WIDTH_90 = 90;

	private Composite workArea;
	private GridData gdTitle;
	private Color BACKGROUND_COLOR = new Color(Display.getDefault(), new RGB(34, 65, 139));
	private IWizard currentWizard;
	private ArrayList<IWizard> createdSubWizards = new ArrayList<>();
	private ArrayList<IWizard> nestedSubWizards = new ArrayList<>();
	private IWizardPage showingPage = null;

	private long runningOperationActivate = 0;
	private long lastJobFinishedTime = LAST_JOB_FINISH_FLAG;

	private String messageOnPage;
	private String descriptionOnPage;

	private int messageTypeOnPage = IMessageProvider.NONE;
	private int pageWidth = SWT.DEFAULT;
	private int pageHeight = SWT.DEFAULT;

	private ProgressMonitorPart progressMonitor;
	private Cursor cursorWaiting;
	private Cursor arrowCursor;

	private MessageDialog closeDialog;
	private boolean useProgressMonitor = true;
	private boolean isMovingToPreviousPage = false;
	private boolean uiLockedFlag = false;

	private Composite buttonBarContainer;

	private Button btnFinish;
	private Button btnNext;
	private Button btnBack;

	private Button btnUninstall;
	private Button btnReinstall;
	private Button btnCancel;
	private Button btnHelp;

	private SelectionAdapter listenerCancel;

	private Composite containerPages;
	private MIWIndowPageFillLayout pageLayout;

	private ListenerList<IPageChangedListener> pageChangedListeners = new ListenerList<>();
	private ListenerList<IPageChangingListener> pageChangingListeners = new ListenerList<>();
	private Button btnShowOnStartUp;
	private static boolean IS_OPEN = false;

	public MIWindowDialog(Shell parentShell, IWizard newWizard, String titleString) {
		super(parentShell);
		pageLayout = new MIWIndowPageFillLayout(5, 5, DEF_MIN_WIDTH, DEF_MIN_HEIGHT);
		setShellStyle(SWT.APPLICATION_MODAL | SWT.CLOSE | SWT.MAX | SWT.TITLE | SWT.BORDER | SWT.MODELESS | SWT.RESIZE
				| getDefaultOrientation());

		setCurrentWizardPage(newWizard);
		listenerCancel = new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				cancelPressed();
			}
		};
	}

	private Object showInProgress(boolean btnCancelEnable) {
		Map<String, Object> savedState = null;

		if (getShell() != null) {
			boolean reqMonitorWithProgres = currentWizard.needsProgressMonitor();
			Display display = getShell().getDisplay();

			cursorWaiting = new Cursor(display, SWT.CURSOR_WAIT);
			setDisplayCursor(cursorWaiting);

			if (useProgressMonitor) {
				btnCancel.removeSelectionListener(listenerCancel);
				arrowCursor = new Cursor(display, SWT.CURSOR_ARROW);
				btnCancel.setCursor(arrowCursor);
			}

			Control currentFocus = display.getFocusControl();
			if (currentFocus != null && currentFocus.getShell() != getShell()) {
				currentFocus = null;
			}

			savedState = saveButtonsState(useProgressMonitor && reqMonitorWithProgres && btnCancelEnable);
			if (currentFocus != null) {
				savedState.put(FOCUS_CONTROL, currentFocus);
			}

			if (reqMonitorWithProgres) {
				if (useProgressMonitor || btnCancelEnable) {
					progressMonitor.attachToCancelComponent(btnCancel);
				}
				progressMonitor.setVisible(true);
			}
			if (lastJobFinishedTime == LAST_JOB_FINISH_FLAG) {
				lastJobFinishedTime = 0;
				getShell().addTraverseListener(e -> {
					final int detail = e.detail;
					final int keyCode = e.keyCode;
					if (detail == SWT.TRAVERSE_RETURN || (detail == SWT.TRAVERSE_MNEMONIC && keyCode == 32)) {
						if (lastJobFinishedTime != 0
								&& System.currentTimeMillis() - lastJobFinishedTime < RESTORE_DELAY) {
							e.doit = false;
							return;
						}
						lastJobFinishedTime = 0;
					}
				});
			}
		}
		return savedState;
	}

	/**
	 * The Back button has been pressed.
	 */
	protected void backPressed() {
		IWizardPage page = showingPage.getPreviousPage();
		if (page == null) {
			// should never happen since we have already visited the page
			return;
		}

		// set flag to indicate that we are moving back
		isMovingToPreviousPage = true;
		// show the page
		showPage(page);
	}

	@Override
	protected void buttonPressed(int idBtn) {
		if (IDialogConstants.HELP_ID == idBtn) {
			helpPressed();
		} else if (IDialogConstants.NEXT_ID == idBtn) {
			nextPressed();
		} else if (IDialogConstants.FINISH_ID == idBtn) {
			finishPressed();
		} else if (IDialogConstants.BACK_ID == idBtn) {
			backPressed();
		}
	}

	@Override
	protected void cancelPressed() {
		if (runningOperationActivate > 0) {
			btnCancel.setEnabled(true);
		} else {
			setReturnCode(CANCEL);
			close();
		}
	}

	@Override
	public boolean close() {
		boolean answer = false;
		if (closeOperation()) {
			answer = hardClose();
		}
		if (answer) {
			IS_OPEN = false;
		}
		return answer;
	}

	@Override
	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		// Register help listener on the shell
		newShell.addHelpListener(event -> {
			// call perform help on the current page
			if (showingPage != null) {
				showingPage.performHelp();
			}
		});
	}

	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		buttonBarContainer = parent;

		((GridLayout) parent.getLayout()).makeColumnsEqualWidth = false;
		btnShowOnStartUp = createShowStartUpButton(parent);

		if (currentWizard.isHelpAvailable()) {
			btnHelp = createButton(parent, IDialogConstants.HELP_ID, IDialogConstants.HELP_LABEL, false);
		}

		btnBack = createButton(parent, IDialogConstants.BACK_ID, IDialogConstants.BACK_LABEL, false);

		btnNext = createButton(parent, IDialogConstants.NEXT_ID, IDialogConstants.NEXT_LABEL, false);
		resizeButton(btnNext, BUTTON_WIDTH_190);

		btnReinstall = createButton(parent, 2001, MIMessages.MIWindowDialog_Reinstall, true);
		resizeButton(btnReinstall, BUTTON_WIDTH_190);

		btnUninstall = createButton(parent, 2000, MIMessages.MIWizardPackPage_UninstallName, true);
		resizeButton(btnUninstall, BUTTON_WIDTH_190);

		btnFinish = createButton(parent, IDialogConstants.FINISH_ID, MIMessages.MIWindowDialog_InstallUpdate, true);

		btnCancel = createBtnCancel(parent);

		toggleButtonForced(btnBack, false);
		toggleButtonForced(btnFinish, false);
		btnNext.setText(MIMessages.MIWindowDialog_InstallUpdate);

	}

	private void resizeButton(Button button, int width) {
		GridData gd = (GridData) button.getLayoutData();
		gd.widthHint = width;
	}

	private void toggleButton(Button button, Boolean visible) {
		boolean isVisible = button.isVisible();
		if (isVisible == visible) {
			return;
		}
		toggleButtonForced(button, visible);
	}

	private void toggleButtonForced(Button button, Boolean visible) {
		GridData data = (GridData) button.getLayoutData();
		data.exclude = !visible;
		if (!visible) {
			((GridLayout) button.getParent().getLayout()).numColumns--;
		} else {
			((GridLayout) button.getParent().getLayout()).numColumns++;
		}
		button.setVisible(visible);
	}

	public void updateButtonsCounters(int installUpdateCounter, int uninstallCounter) {
		btnNext.setText(
				(installUpdateCounter > 0)
						? (MIMessages.MIWindowDialog_InstallUpdate + " " + installUpdateCounter + " "
								+ MIMessages.MIWizardPackPage_ItemsLabel)
						: MIMessages.MIWindowDialog_InstallUpdate);
		btnUninstall
				.setText((uninstallCounter > 0)
						? (MIMessages.MIWizardPackPage_UninstallName + " " + uninstallCounter + " "
								+ MIMessages.MIWizardPackPage_ItemsLabel)
						: MIMessages.MIWizardPackPage_UninstallName);
	}

	public void setPackPageButtonbar() {
		toggleButton(btnBack, false);
		toggleButton(btnFinish, false);
		toggleButton(btnReinstall, true);
		toggleButton(btnUninstall, true);
		toggleButton(btnNext, true);
		resizeButton(btnNext, BUTTON_WIDTH_190);
		btnNext.setText(MIMessages.MIWindowDialog_InstallUpdate);
		buttonBarContainer.layout();
		buttonBarContainer.getParent().layout();
	}

	public void setSumPageButtonbar() {
		toggleButton(btnBack, true);
		toggleButton(btnNext, true);
		resizeButton(btnNext, BUTTON_WIDTH_90);
		btnNext.setText(IDialogConstants.NEXT_LABEL);
		toggleButton(btnReinstall, false);
		toggleButton(btnUninstall, false);
		toggleButton(btnFinish, true);
		btnFinish.setText(IDialogConstants.FINISH_LABEL);
		buttonBarContainer.layout();
		buttonBarContainer.getParent().layout();
	}

	public void setLicensePageButtonbar() {
		toggleButton(btnBack, true);
		toggleButton(btnNext, false);
		toggleButton(btnReinstall, false);
		toggleButton(btnUninstall, false);
		toggleButton(btnFinish, true);
		btnFinish.setText(IDialogConstants.FINISH_LABEL);
		buttonBarContainer.layout();
		buttonBarContainer.getParent().layout();
	}

	private Button createShowStartUpButton(Composite parent) {
		Layout layout = parent.getLayout();
		if (layout instanceof GridLayout) {
			GridLayout gridLayout = (GridLayout) layout;
			gridLayout.numColumns++;
		}

		Preferences pref = ModularInstallerUIActivator.getInstance().getPreferenceStoreForModularPreferencePage();
		Button button = new Button(parent, SWT.CHECK);
		button.setText(MIMessages.MIWindowDialog_ShowOnStartUp);
		button.setSelection(pref.getBoolean(ModularPreferenceConstants.PREF_MI_SHOW_ON_START_UP, false));
		button.setFont(JFaceResources.getDialogFont());
		button.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent event) {
				boolean booleanValue = ((Button) event.getSource()).getSelection();

				pref.putBoolean(ModularPreferenceConstants.PREF_MI_SHOW_ON_START_UP, booleanValue);
			}
		});
		setButtonLayoutData(button);
		return button;

	}

	private Point calcPageDeltaSize(IWizardPage page) {
		final Point point = new Point(0, 0);
		Control pageControl = page.getControl();
		if (pageControl == null) {
			return point;
		}
		Point contentSize = pageControl.computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
		Rectangle rect = pageLayout.getAreaRectangle(containerPages);
		Point containerSize = new Point(rect.width, rect.height);
		final int maxByX = Math.max(0, contentSize.x - containerSize.x);
		final int maxByY = Math.max(0, contentSize.y - containerSize.y);
		return new Point(maxByX, maxByY);
	}

	public void hideActionButtons() {
		Composite cmpButtonsBar = (Composite) buttonBar;
		hideButtonsOnComposite(cmpButtonsBar);
		btnUninstall.dispose();
		btnCancel.setVisible(true);

	}

	public void invokeUninstall() {
		btnUninstall.notifyListeners(SWT.Selection, new Event());
	}

	public void invokeReinstall() {
		btnReinstall.notifyListeners(SWT.Selection, new Event());
	}

	private void hideButtonsOnComposite(Composite cmpButtonsBar) {
		for (Control item : cmpButtonsBar.getChildren()) {
			if (item instanceof Button) {
				((Button) item).setVisible(false);
			}
			if (item instanceof Composite) {
				hideButtonsOnComposite((Composite) item);
			}
		}
	}

	public void showUninstallButton(boolean show) {
		if (btnUninstall != null && !btnUninstall.isDisposed()) {
			btnUninstall.setEnabled(show);
		}
	}

	public void selectOnStartUpButton(boolean show) {
		if (btnShowOnStartUp != null && !btnShowOnStartUp.isDisposed()) {
			btnShowOnStartUp.setSelection(show);
		}
	}

	public void addUninstallButtonListener(SelectionListener listener) {
		if (btnUninstall != null && !btnUninstall.isDisposed()) {
			btnUninstall.addSelectionListener(listener);
		}
	}

	public void showReinstallButton(boolean show) {
		if (btnReinstall != null && !btnReinstall.isDisposed()) {
			btnReinstall.setEnabled(show);
		}
	}

	public void addReinstallButtonListener(SelectionListener listener) {
		if (btnReinstall != null && !btnReinstall.isDisposed()) {
			btnReinstall.addSelectionListener(listener);
		}
	}

	@Override
	protected void setButtonLayoutData(Button button) {
		GridData dataButton = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		int hintbyWidth = convertHorizontalDLUsToPixels(IDialogConstants.BUTTON_WIDTH);
		final int btnMinWidht = button.getDisplay().getBounds().width / 5;
		Point minSize = button.computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
		dataButton.widthHint = Math.max(Math.min(hintbyWidth, btnMinWidht), minSize.x);
		button.setLayoutData(dataButton);
	}

	private Button createBtnCancel(Composite parent) {
		GridLayout layout = (GridLayout) parent.getLayout();
		layout.numColumns++;
		Button btnCancel = new Button(parent, SWT.PUSH);
		btnCancel.setFont(parent.getFont());
		btnCancel.setText(IDialogConstants.CANCEL_LABEL);
		btnCancel.setData(Integer.valueOf(IDialogConstants.CANCEL_ID));
		setButtonLayoutData(btnCancel);
		btnCancel.addSelectionListener(listenerCancel);
		return btnCancel;
	}

	@Override
	protected Button getButton(int id) {
		if (id == IDialogConstants.CANCEL_ID) {
			return btnCancel;
		}
		return super.getButton(id);
	}

	@Override
	protected Control createDialogArea(Composite parent) {
		Composite composite = (Composite) super.createDialogArea(parent);
		composite.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WHITE));

		// Build the Page container
		containerPages = createPageContainer(composite);
		GridData gridDataDef = new GridData(GridData.FILL_BOTH);
		gridDataDef.widthHint = pageWidth;
		gridDataDef.heightHint = pageHeight;

		containerPages.setLayoutData(gridDataDef);
		containerPages.setFont(parent.getFont());
		containerPages.setBackground(MIWizardUtil.CLR_BG_WHITE);
		Label separatoseparatorLabelrBottom1 = new Label(composite, SWT.HORIZONTAL | SWT.SEPARATOR);
		separatoseparatorLabelrBottom1.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_WHITE));
		separatoseparatorLabelrBottom1.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		// Insert a progress monitor
		progressMonitor = createMonitorWithProgress(composite, new GridLayout());
		progressMonitor.setBackground(MIWizardUtil.CLR_BG_WHITE);
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		if (!currentWizard.needsProgressMonitor()) {
			gridData.exclude = true;
		}

		progressMonitor.setLayoutData(gridData);
		for (Control iter : progressMonitor.getChildren()) {
			iter.setBackground(MIWizardUtil.CLR_BG_WHITE);
		}
		progressMonitor.setVisible(false);
		// Build the separator line
		Label separatorLabel = new Label(composite, SWT.HORIZONTAL | SWT.SEPARATOR);
		separatorLabel.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		applyDialogFont(progressMonitor);
		return composite;
	}

	private Composite createPageContainer(Composite parent) {
		Composite result = new Composite(parent, SWT.NONE);
		result.setLayout(MIGridLayoutFactory.createGridlayout(0, 0, 1, false));
		result.setLayout(pageLayout);
		return result;
	}

	private void createPageControls() {
		// Allow the wizard pages to precreate their page controls
		// This allows the wizard to open to the correct size
		currentWizard.createPageControls(containerPages);
		// Ensure that all of the created pages are initially not visible
		IWizardPage[] pages = currentWizard.getPages();
		for (int i = 0; i < pages.length; i++) {
			IWizardPage page = pages[i];
			if (page.getControl() != null) {
				page.getControl().setVisible(false);
			}
		}
	}

	private MessageDialog getCloseMessageDialog() {
		final String dialogTitle = JFaceResources.getString("WizardClosingDialog.title"); //$NON-NLS-1$
		final String dialogMessage = JFaceResources.getString("WizardClosingDialog.message"); //$NON-NLS-1$

		MessageDialog dialog = new MessageDialog(getShell(), dialogTitle, null, dialogMessage, MessageDialog.QUESTION,
				0, IDialogConstants.OK_LABEL) {
			@Override
			protected int getShellStyle() {
				return super.getShellStyle() | SWT.SHEET;
			}
		};
		return dialog;
	}

	/**
	 * The Finish button has been pressed.
	 */
	protected void finishPressed() {
		if (currentWizard.performFinish()) {
			// Call perform finish on outer wizards in the nested chain
			// (to allow them to save state for example)
			for (int i = 0; i < nestedSubWizards.size() - 1; i++) {
				nestedSubWizards.get(i).performFinish();
			}
			// Hard close the dialog.
			setReturnCode(OK);
			hardClose();
		}
	}

	@Override
	public IWizardPage getCurrentPage() {
		return showingPage;
	}

	protected IProgressMonitor getProgressMonitor() {
		return progressMonitor;
	}

	protected IWizard getWizard() {
		return currentWizard;
	}

	private boolean hardClose() {
		for (int i = 0; i < createdSubWizards.size(); i++) {
			IWizard createdWizard = createdSubWizards.get(i);
			try {
				createdWizard.dispose();
			} catch (Exception e) {
				Status status = new Status(IStatus.ERROR, Policy.JFACE, IStatus.ERROR, e.getMessage(), e);
				Policy.getLog().log(status);
			}
			createdWizard.setContainer(null);
		}
		setTitleImage(null);
		return super.close();
	}

	/**
	 * The Help button has been pressed.
	 */
	protected void helpPressed() {
		if (showingPage != null) {
			showingPage.performHelp();
		}
	}

	protected void nextPressed() {
		IWizardPage page = showingPage.getNextPage();
		if (page != null) {
			showPage(page);
		}
	}

	private boolean doPageChanging(IWizardPage targetPage) {
		PageChangingEvent event = new PageChangingEvent(this, getCurrentPage(), targetPage);
		firePageChanging(event);
		return event.doit;
	}

	private boolean closeOperation() {
		if (runningOperationActivate > 0) {
			synchronized (this) {
				closeDialog = getCloseMessageDialog();
			}
			closeDialog.open();
			synchronized (this) {
				closeDialog = null;
			}
			return false;
		}
		return currentWizard.performCancel();
	}

	private void restoreStateEanble(Control control, Map<String, Object> saveState, String key) {
		if (control != null) {
			Boolean b = (Boolean) saveState.get(key);
			if (b != null) {
				control.setEnabled(b.booleanValue());
			}
		}
	}

	private void restoreUIState(Map<String, Object> saveState) {
		restoreStateEanble(btnNext, saveState, "next"); //$NON-NLS-1$
		restoreStateEanble(btnHelp, saveState, "help"); //$NON-NLS-1$
		restoreStateEanble(btnFinish, saveState, "finish"); //$NON-NLS-1$
		restoreStateEanble(btnCancel, saveState, "cancel"); //$NON-NLS-1$
		restoreStateEanble(btnBack, saveState, "back"); //$NON-NLS-1$
		Object pageValue = saveState.get("page"); //$NON-NLS-1$
		if (pageValue != null) {
			((ControlEnableState) pageValue).restore();
		}
	}

	@Override
	public void run(boolean fork, boolean cancelable, IRunnableWithProgress runnable)
			throws InvocationTargetException, InterruptedException {
		Object state = null;
		IProgressMonitor monitor = getProgressMonitor();
		if (monitor == null) {
			monitor = new NullProgressMonitor();
		}

		if (runningOperationActivate++ == 0) {
			state = showInProgress(fork && cancelable);
		}
		try {
			if (!fork) {
				uiLockedFlag = true;
			}
			ModalContext.run(runnable, fork, monitor, getShell().getDisplay());
			uiLockedFlag = false;
		} finally {
			monitor.done();
			if (state != null) {
				lastJobFinishedTime = System.currentTimeMillis();
				stopped(state);
			}
			runningOperationActivate--;
		}
	}

	private void saveButtonState(Button button, Map<String, Object> saveState, String key, boolean enabled) {
		if (button != null) {
			saveState.put(key, button.getEnabled() ? Boolean.TRUE : Boolean.FALSE);
			button.setEnabled(enabled);
		}
	}

	private Map<String, Object> saveButtonsState(boolean isCancelEnabled) {
		Map<String, Object> savedState = new HashMap<>(10);
		saveButtonState(btnBack, savedState, "back", false); //$NON-NLS-1$
		saveButtonState(btnNext, savedState, "next", false); //$NON-NLS-1$
		saveButtonState(btnFinish, savedState, "finish", false); //$NON-NLS-1$
		saveButtonState(btnCancel, savedState, "cancel", isCancelEnabled); //$NON-NLS-1$
		saveButtonState(btnUninstall, savedState, "uninstall", false); //$NON-NLS-1$
		saveButtonState(btnReinstall, savedState, "reinstall", false); //$NON-NLS-1$
		saveButtonState(btnHelp, savedState, "help", false); //$NON-NLS-1$
		if (showingPage != null) {
			savedState.put("page", ControlEnableState.disable(showingPage.getControl())); //$NON-NLS-1$
		}
		return savedState;
	}

	private void setDisplayCursor(Cursor c) {
		Shell[] shells = getShell().getDisplay().getShells();
		for (int i = 0; i < shells.length; i++) {
			shells[i].setCursor(c);
		}
	}

	public void setPageSize(int width, int height) {
		pageWidth = width;
		pageHeight = height;
	}

	public void setPageSize(Point size) {
		setPageSize(size.x, size.y);
	}

	protected void setCurrentWizardPage(IWizard newWizard) {
		currentWizard = newWizard;
		currentWizard.setContainer(this);

		if (!createdSubWizards.contains(currentWizard)) {
			nestedSubWizards.add(currentWizard);
			createdSubWizards.add(currentWizard);
			if (containerPages != null) {
				createPageControls();
				updateSizeForWizard(currentWizard);
				containerPages.layout(true);
			}
		} else {
			int size = nestedSubWizards.size();
			if (size >= 2 && nestedSubWizards.get(size - 2) == currentWizard) {
				nestedSubWizards.remove(size - 1);
			} else {
				nestedSubWizards.add(currentWizard);
			}
		}
	}

	private void updateForPage(IWizardPage wizardPage) {
		if (currentWizard != wizardPage.getWizard()) {
			setCurrentWizardPage(wizardPage.getWizard());
		}

		if (wizardPage.getControl() == null) {
			wizardPage.createControl(containerPages);
			final String msg = JFaceResources.getString("WizardDialog.missingSetControl"); //$NON-NLS-1$
			Assert.isNotNull(wizardPage.getControl(), JFaceResources.format(msg, wizardPage.getName()));

			((Composite) dialogArea).layout();
			// updateSizeforWizardPage(wizardPage);
		}
		// make the new page visible
		IWizardPage oldPage = showingPage;
		showingPage = wizardPage;

		if (oldPage != null) {
			oldPage.setVisible(false);
		}
		showingPage.setVisible(true);
		updateWizadDialog();
	}

	/**
	 * Shows the starting page of the wizard.
	 */
	private void showStartingPage() {
		showingPage = currentWizard.getStartingPage();
		if (showingPage != null) {
			final Control pageControl = showingPage.getControl();
			if (pageControl == null) {
				showingPage.createControl(containerPages);
				Assert.isNotNull(pageControl);
			}
			showingPage.setVisible(true);
			updateWizadDialog();
		}
	}

	private void stopped(Object savedState) {
		if (getShell() != null && !getShell().isDisposed()) {
			if (currentWizard.needsProgressMonitor()) {
				progressMonitor.setVisible(false);
				progressMonitor.removeFromCancelComponent(btnCancel);
			}

			@SuppressWarnings("unchecked")
			Map<String, Object> state = (Map<String, Object>) savedState;
			restoreUIState(state);
			setDisplayCursor(null);
			if (useProgressMonitor) {
				btnCancel.addSelectionListener(listenerCancel);
				btnCancel.setCursor(null);
				arrowCursor.dispose();
				arrowCursor = null;
			}
			cursorWaiting.dispose();
			cursorWaiting = null;
			Control focusControl = (Control) state.get(FOCUS_CONTROL);
			if (focusControl != null && !focusControl.isDisposed()) {
				focusControl.setFocus();
			}
		}
	}

	@Override
	public void showPage(IWizardPage page) {
		if (page == null) {
			return;
		}
		if (page == showingPage) {
			return;
		}

		if (isMovingToPreviousPage) {
			isMovingToPreviousPage = false;
		} else {
			page.setPreviousPage(showingPage);
		}

		if (!doPageChanging(page))
			return;

		// Update for the new page in a busy cursor if possible
		if (getContents() == null) {
			updateForPage(page);
		} else {
			final IWizardPage finalPage = page;
			BusyIndicator.showWhile(getContents().getDisplay(), () -> updateForPage(finalPage));
		}
	}

	protected void updateWizadDialog() {
		updateWindowTitle();
		updateTitleBar();
		updateButtons();
		final PageChangedEvent event = new PageChangedEvent(this, getCurrentPage());
		firePageChanged(event);
	}

	@Override
	public void updateButtons() {
		boolean canFlipToNextPage = false;
		boolean canFinish = currentWizard.canFinish();
		if (btnBack != null) {
			boolean backEnabled = showingPage != null && showingPage.getPreviousPage() != null;
			btnBack.setEnabled(backEnabled);
		}
		if (btnNext != null) {
			canFlipToNextPage = showingPage != null && showingPage.canFlipToNextPage();
			btnNext.setEnabled(canFlipToNextPage);
		}
		btnFinish.setEnabled(canFinish);
		// finish is default unless it is disabled and next is enabled
		if (canFlipToNextPage && !canFinish) {
			getShell().setDefaultButton(btnNext);
		} else {
			getShell().setDefaultButton(btnFinish);
		}
	}

	@Override
	public void updateMessage() {

		if (showingPage == null) {
			return;
		}

		messageOnPage = showingPage.getMessage();
		if (messageOnPage != null && showingPage instanceof IMessageProvider) {
			messageTypeOnPage = ((IMessageProvider) showingPage).getMessageType();
		} else {
			messageTypeOnPage = IMessageProvider.NONE;
		}
		if (messageOnPage == null) {
			setMessage(descriptionOnPage);
		} else {
			setMessage(messageOnPage, messageTypeOnPage);
		}
		setErrorMessage(showingPage.getErrorMessage());
	}

	private void setShellSize(int width, int height) {
		Rectangle size = getShell().getBounds();
		size.height = height;
		size.width = width;
		getShell().setBounds(getConstrainedShellBounds(size));
	}

	protected void updateSizeforWizardPage(IWizardPage wizardPage) {
		if (wizardPage == null || wizardPage.getControl() == null) {
			return;
		}
		Shell shell = getShell();
		Point shellSize = shell.getSize();
		Point delta = calcPageDeltaSize(wizardPage);
		if (delta.x > 0 || delta.y > 0) {
			setShellSize(shellSize.x + delta.x, shellSize.y + delta.y);
			constrainShellSize();
		}
		pageLayout.setLayoutPageControls(wizardPage.getControl());
	}

	@Override
	public void updateSize() {
		updateSizeforWizardPage(showingPage);
	}

	private void updateSizeForWizard(IWizard sizingWizard) {
		Point delta = new Point(0, 0);
		IWizardPage[] pages = sizingWizard.getPages();
		for (int i = 0; i < pages.length; i++) {

			Point pageDelta = calcPageDeltaSize(pages[i]);
			delta.x = Math.max(delta.x, pageDelta.x);
			delta.y = Math.max(delta.y, pageDelta.y);
		}
		if (delta.x > 0 || delta.y > 0) {

			Shell shell = getShell();
			Point shellSize = shell.getSize();
			setShellSize(shellSize.x + delta.x, shellSize.y + delta.y);
		}
	}

	public static boolean isOpen() {
		return IS_OPEN;
	}

	@Override
	public void updateTitleBar() {
		String showingTitle = ""; //$NON-NLS-1$
		if (showingPage != null && showingPage.getTitle() != null) {
			showingTitle = showingPage.getTitle();
		}

		setTitle(showingTitle);
		if (showingPage != null) {
			setTitleImage(showingPage.getImage());
		}
		updateMessage();
	}

	@Override
	public void updateWindowTitle() {
		final Shell shell = getShell();
		String title = currentWizard.getWindowTitle();
		if (shell != null && title != null) {
			shell.setText(title);
		}
		if (title == null) {
			title = ""; //$NON-NLS-1$
		}
	}

	@Override
	public Object getSelectedPage() {
		return getCurrentPage();
	}

	@Override
	public void addPageChangedListener(IPageChangedListener listener) {
		pageChangedListeners.add(listener);
	}

	@Override
	public void removePageChangedListener(IPageChangedListener listener) {
		pageChangedListeners.remove(listener);
	}

	protected void firePageChanged(final PageChangedEvent event) {
		Object[] listeners = pageChangedListeners.getListeners();
		for (int i = 0; i < listeners.length; ++i) {
			final IPageChangedListener l = (IPageChangedListener) listeners[i];
			SafeRunnable.run(new SafeRunnable() {
				@Override
				public void run() {
					l.pageChanged(event);
				}
			});
		}
	}

	public void addPageChangingListener(IPageChangingListener listener) {
		pageChangingListeners.add(listener);
	}

	public void removePageChangingListener(IPageChangingListener listener) {
		pageChangingListeners.remove(listener);
	}

	protected void firePageChanging(final PageChangingEvent event) {
		if (event.getTargetPage() instanceof MIWizardPackPage) {
			setPackPageButtonbar();
		}

		if (event.getTargetPage() instanceof MIWizardSumPage) {
			setSumPageButtonbar();
		}

		if (event.getTargetPage() instanceof AcceptLicensesWizardPage) {
			setLicensePageButtonbar();
		}
		Object[] listeners = pageChangingListeners.getListeners();
		for (int i = 0; i < listeners.length; ++i) {
			final IPageChangingListener l = (IPageChangingListener) listeners[i];
			SafeRunnable.run(new SafeRunnable() {
				@Override
				public void run() {
					l.handlePageChanging(event);
				}
			});
		}
	}

	@Override
	protected Control createContents(Composite parent) {
		currentWizard.addPages();
		Composite contents = new Composite(parent, SWT.NONE);
		contents.setLayoutData(new GridData(GridData.FILL_BOTH));
		initializeDialogUnits(contents);
		contents.setLayout(MIGridLayoutFactory.createGridlayout(0, 0, 1, false));
		createTitleArea(contents);

		workArea = new Composite(contents, SWT.BORDER);
		GridLayout childLayout = MIGridLayoutFactory.createGridlayout(0, 0, 1, false);
		workArea.setLayout(childLayout);
		workArea.setLayoutData(new GridData(GridData.FILL_BOTH));
		workArea.setFont(JFaceResources.getDialogFont());
		initializeDialogUnits(workArea);
		dialogArea = createDialogArea(workArea);
		buttonBar = createButtonBar(workArea);

		createPageControls();
		showStartingPage();
		IS_OPEN = true;
		return contents;
	}

	private Control createTitleArea(Composite parent) {
		gdTitle = new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1);
		gdTitle.heightHint = 64;

		GridData gdlabelText = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);

		Composite cmpsTitle = new Composite(parent, SWT.NONE);
		cmpsTitle.setBackground(BACKGROUND_COLOR);
		cmpsTitle.setLayoutData(gdTitle);
		cmpsTitle.setLayout(MIGridLayoutFactory.createGridlayout(0, 0, 1, false));

		Composite linkContainer = new Composite(cmpsTitle, SWT.NONE);
		linkContainer.setLayoutData(gdlabelText);
		linkContainer.setLayout(MIGridLayoutFactory.createGridlayout(13, 0, 1, false));
		linkContainer
				.setBackgroundImage(ModularInstallerUIActivator.getInstance().getImage(MIRegistryImages.IMG_HEAD_LB));

		return cmpsTitle;
	}

	protected ProgressMonitorPart createMonitorWithProgress(Composite composite, GridLayout pmlayout) {
		useProgressMonitor = false;
		return new ProgressMonitorPart(composite, pmlayout, true) {
			String curTask = null;

			@Override
			public void clearBlocked() {
				super.clearBlocked();
				if (!uiLockedFlag) {
					getBlockedHandler().clearBlocked();
				}
			}

			@Override
			public void setBlocked(IStatus reason) {
				super.setBlocked(reason);
				if (!uiLockedFlag) {
					getBlockedHandler().showBlocked(getShell(), this, reason, curTask);
				}
			}

			@Override
			public void setTaskName(String name) {
				super.setTaskName(name);
				curTask = name;
			}

			@Override
			public void subTask(String name) {
				super.subTask(name);
				// If we haven't got anything yet use this value for more
				// context
				if (curTask == null) {
					curTask = name;
				}
			}

			@Override
			public void beginTask(String name, int totalWork) {
				super.beginTask(name, totalWork);
				curTask = name;
			}
		};
	}

}
