/*******************************************************************************
 * Copyright (c) 2014 Freescale Semiconductor, Inc
 * Copyright 2023 NXP
 * 
 * SPDX-License-Identifier: EPL-1.0
 *******************************************************************************/

package com.nxp.sa.ui.common.epl;

import java.net.URL;
import java.util.ArrayList;

import com.freescale.sa.util.Logger;
import com.freescale.sa.util.LoggerStatus;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

//import com.freescale.sa.ui.common.logger.OS;
import com.freescale.sa.util.CommonConstants;

/**
 * The activator class controls the plug-in life cycle
 */
public class SAUILSPlugin extends AbstractUIPlugin {

	// The plug-in ID
	public static final String PLUGIN_ID = "com.freescale.sa.ui.common"; //$NON-NLS-1$

	// The shared instance
	private static SAUILSPlugin plugin;
	
	// the SA Logger
	private static Logger LOGGER;
	
	/**
	 * Status code indicating an unexpected internal error.
	 */
	public static final int INTERNAL_ERROR = 120;
	
	/**
	 * The constructor
	 */
	public SAUILSPlugin() {
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.plugin.AbstractUIPlugin#start(org.osgi.framework.BundleContext)
	 */
	public void start(BundleContext context) throws Exception {
		super.start(context);
		LOGGER = Logger.getLogger(context.getBundle());
		plugin = this;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.plugin.AbstractUIPlugin#stop(org.osgi.framework.BundleContext)
	 */
	public void stop(BundleContext context) throws Exception {
		plugin = null;
		LOGGER = null;
		super.stop(context);
	}
	
	/**
	 * Returns the shared instance
	 *
	 * @return the shared instance
	 */
	public static SAUILSPlugin getInstance() {
		return plugin;
	}
	
	/**
	 * Log as error Issues and logs an error with an exception.
	 * 
	 * @param msg - a description of the error condition.
	 * @param e - and exception. Maybe null if no exception is available
	 * @param s - status containing details about the failure to be logged.
	 */
	
	/**
	 * Returns the instance of the logger
	 * 
	 * @since 2.0
	 */

	public Logger getLogger() {
		return LOGGER;
	}
	
	public void logErrorStatus(String msg, Throwable e, IStatus s) {
		if (getInstance().getLog() != null) {
			IStatus status;
			if (s != null) {
				status = new MultiStatus(PLUGIN_ID, 0, new IStatus[] { s }, msg, e);
			} else {
				status = new Status(IStatus.ERROR, PLUGIN_ID, IStatus.OK, msg, e);
			}

			getInstance().getLog().log(status);
		}
		
		if (LOGGER != null) {
			logError(msg, e);
		}
	}

	/**
	 * Logs an internal error with the specified message.
	 *
	 * @param message
	 *            the error message to log
	 */
	public void logError(String message) {
		LoggerStatus status = getLogger().getErrorStatus(message);
		getLogger().log(status);
	}
	
	public void logError(String msg, Throwable exc) {
		LoggerStatus status = getLogger().getErrorStatus(msg, exc);
		getLogger().log(status);
	}

	/**
	 * Logs an internal error with the specified message.
	 *
	 * @param message
	 *            the error message to log
	 */
	public void logInfo(String message) {
		LoggerStatus status = getLogger().getInfoStatus(message);
		getLogger().log(status);
	}
	
	public void logInfo(String msg, Throwable exc) {
		LoggerStatus status = getLogger().getInfoStatus(msg, exc);
		getLogger().log(status);
	}

	
	/**
	 * Logs the specified status with this plug-in's log.
	 * 
	 * @param status status to log
	 */
	public void log(LoggerStatus status) {
		getDefault().getLogger().log(status);
	}
	
	/**
	 * Logs the specified throwable with this plug-in's log.
	 * 
	 * @param t throwable to log 
	 */
	public void log(Throwable t) {
		LoggerStatus status = getLogger().getErrorStatus("Error logged from SA Plugin: ", t); //$NON-NLS-1$

		log(status);
	}
	
	/**
	 * Convenience method which returns the unique identifier of this plug-in.
	 * 
	 * @return debug plug-in identifier
	 */
	public static String getUniqueIdentifier() {
		return PLUGIN_ID;
	}
	
	/**
	 * Returns the shared instance
	 *
	 * @return the shared instance
	 */
	public static SAUILSPlugin getDefault() {
		return plugin;
	}
	
	/**
	 * Returns an image descriptor for the image file at the given plug-in
	 * relative path
	 * 
	 * @param path
	 *            the path
	 * @return the image descriptor
	 */
	public static ImageDescriptor getImageDescriptor(String path) {
		return imageDescriptorFromPlugin(PLUGIN_ID, path);
	}
	
	public String getLocation() {
		URL pluginURL = FileLocator.find(plugin.getBundle(), new Path(CommonConstants.EMPTY_STRING),
				null);
		if (pluginURL != null) {
			try {
				URL localURL = FileLocator.toFileURL(FileLocator
						.resolve(pluginURL));
				String path = localURL.getPath();
				if (OS.isWindows() && path.charAt(0) == '/') {
					path = path.substring(1);
				}
				return path;
			} catch (Exception ex) {
				logError( "Failed to get plugin location: ", ex);
				return null;
			}
		} else {
			logError("Failed to get plugin location with a null pluginURL.");
			return null;
		}
	}

	/**
	 * Get all the SA opened editors that match the given baseName
	 * @param baseName The results files basename
	 * @return Array of opened SA editors.
	 */
	public static AbstractSaEditor[] getOpenedEditorsByBaseName(String baseName){
		ArrayList<AbstractSaEditor> saEditorList = new ArrayList<AbstractSaEditor>();
		try {
			// first get search for the editor
			IWorkbenchPage wbPage = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
			if (wbPage == null) {
				// nothing else matters
				return null;
			}
			IEditorReference[] editorsVect = wbPage.getEditorReferences();
			for (IEditorReference ed : editorsVect) {
				IEditorPart edPart = ed.getEditor(false);
				if (edPart instanceof AbstractSaEditor) {
					AbstractSaEditor saEd = (AbstractSaEditor)edPart;
					if (baseName.equals(saEd.getBaseName())) {
						// we just found our editor\
						IEditorPart editor = wbPage.findEditor(saEd.getEditorInput());
						if (editor == null || !(editor instanceof AbstractSaEditor)) {
							continue;
						}
						AbstractSaEditor saEditor = (AbstractSaEditor)editor;
						saEditorList.add(saEditor);								
					}
				} 
			}
		} catch (Exception e1) {
			getDefault().logError("[ExcludeSymbolsDialog:refreshStatistics] Could not refresh editors"); //$NON-NLS-1$
		}
		AbstractSaEditor[] saEditors = new AbstractSaEditor[saEditorList.size()];
		saEditorList.toArray(saEditors);
		return saEditors;
	}

}
