/*******************************************************************************
 * Copyright (c) 2001, 2006 IBM Corporation and others.
 * Copyright 2023 NXP
 * SPDX-License-Identifier: EPL-1.0
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Freescale Semiconductor - modified by and for Freescale
 *******************************************************************************/

package com.freescale.system.browser.epl;

import org.eclipse.swt.SWT;
import org.eclipse.swt.accessibility.ACC;
import org.eclipse.swt.accessibility.Accessible;
import org.eclipse.swt.accessibility.AccessibleAdapter;
import org.eclipse.swt.accessibility.AccessibleControlAdapter;
import org.eclipse.swt.accessibility.AccessibleControlEvent;
import org.eclipse.swt.accessibility.AccessibleEvent;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.MouseTrackAdapter;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.TraverseEvent;
import org.eclipse.swt.events.TraverseListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;

import com.freescale.system.browser2.Messages2;
import com.freescale.system.browser.epl.ITabDescriptorEpl;
import com.freescale.system.browser.epl.TabbedWidgetFactoryEpl;


/**
 * The list of tabs in the tabbed sheet page.
 */
public class TabbedListEpl extends Composite {

	private static final ListElement[] ELEMENTS_EMPTY = new ListElement[0];

	private static final int ANTICIPATE_ICON = 32;

	protected static final int INDENT = 7;

	private boolean fFocus = false;

	private ListElement[] fElements;

	private int fSelectedElementIndex = -1;

	private int fTopVisibleIndex = -1;

	private int fBottomVisibleIndex = -1;

	private TopNavigationElement fTopNavigationElement;

	private BottomNavigationElement fBottomNavigationElement;

	private int fWidestLabelIndex = -1;

	private int fTabsThatFitInComposite = -1;

	private Color fHoverBackground;

	private Color fDefaultBackground;

	private Color fDefaultForeground;

	private Color fActiveBackground;

	private Color fBorder;

	private Color fDarkShadow;

	private Color fTextColor;

	private TabbedWidgetFactoryEpl fFactory;

	/**
	 * Inner class representing just one element. 
	 * Responsible for the actual painting of the tab. 
	 * Associates with mouse an paint listeners.
	 */
	public class ListElement extends Canvas {

		/**
		 * The tab this list element holds.
		 */
		private ITabDescriptorEpl fTab;

		/**
		 * The index of the tab in the list.
		 */
		private int fIndex;

		/**
		 * The selection status of the tab.
		 */
		private boolean fSelected;
		
		/**
		 * The hovering status.
		 */
		private boolean fHover;

		/**
		 * Ctor.
		 * 
		 * @param parent the composite widget that will hold this list element
		 * @param tab the tab (cannot be changed)
		 * @param index the index of the tab in the list
		 */
		public ListElement(Composite parent, final ITabDescriptorEpl tab, int index) {
			super(parent, SWT.NO_FOCUS);
			fTab = tab;
			fHover = false;
			fSelected = false;
			fIndex = index;

			addPaintListener(new PaintListener() {

				public void paintControl(PaintEvent e) {
					paint(e);
				}
			});
			addMouseListener(new MouseAdapter() {

				public void mouseUp(MouseEvent e) {
					if (!fSelected) {
						select(getIndex(ListElement.this), true);
					}
				}
			});
			addMouseMoveListener(new MouseMoveListener() {

				public void mouseMove(MouseEvent e) {
					if (!fHover) {
						fHover = true;
						redraw();
					}
				}
			});
			addMouseTrackListener(new MouseTrackAdapter() {

				public void mouseExit(MouseEvent e) {
					fHover = false;
					redraw();
				}
			});
		}

		/**
		 * Sets the selection status of the tab.
		 * 
		 * @param selected the selection status
		 */
		public void setSelected(boolean selected) {
			this.fSelected = selected;
			redraw();
		}

		/**
		 * Draws elements and collects element areas.
		 */
		private void paint(PaintEvent e) {
			e.gc.setBackground(fDefaultBackground);
			e.gc.setForeground(fDefaultForeground);
			Rectangle bounds = getBounds();

			e.gc.fillRectangle(0, 0, bounds.width, bounds.height);

			// draw tab
			if (fSelected) {
				e.gc.setBackground(fActiveBackground);
			} else if (fHover) {
				e.gc.setBackground(fHoverBackground);
			} else {
				e.gc.setBackground(fDefaultBackground);
			}

			/* draw the fill in the tab */
			if (fSelected) {
				e.gc.fillRectangle(4, 0, bounds.width, bounds.height);
				e.gc.fillRectangle(3, 1, 3, bounds.height);
			} else if (fHover) {
				e.gc.fillRectangle(2, 0, bounds.width - 4, bounds.height);
			}

			/* draw the border */
			if (fSelected) {
				e.gc.setForeground(fBorder);
				e.gc.drawLine(4, 0, bounds.width - 1, 0);
				e.gc.drawPoint(3, 1);
				e.gc.drawPoint(3, bounds.height - 1);
				e.gc.drawLine(2, 2, 2, bounds.height - 2);
			} else {
				e.gc.setForeground(fBorder);
				if (getSelectionIndex() >= 0 && getSelectionIndex() + 1 == fIndex) {
					e.gc.drawLine(4, 0, bounds.width - 1, 0);
				} else {
					e.gc.drawLine(2, 0, bounds.width - 3, 0);
				}
				e.gc.drawLine(bounds.width - 1, 0, bounds.width - 1, bounds.height - 1);
			}

			int textIndent = INDENT;
			FontMetrics fm = e.gc.getFontMetrics();
			int height = fm.getHeight();
			int textMiddle = (bounds.height - height) / 2;

			if (fSelected && fTab.getImage() != null) {

				// draw the icon for the selected tab
				e.gc.drawImage(fTab.getImage().createImage(), textIndent - 2, textMiddle);
				textIndent = textIndent + 16 + 2;
			} else if (fTab.isIndented()) {

				// draw the indent tiny square 
				e.gc.drawRectangle(20, textMiddle + 6, 1, 1);
				textIndent = textIndent + 16 + 4;
			}

			/* draw the text */
			e.gc.setForeground(fTextColor);
			e.gc.drawText(fTab.getLabel(), textIndent, textMiddle);
			if (((TabbedListEpl)getParent()).fFocus && fSelected) {

				/* draw a line if the tab has focus */
				Point point = e.gc.textExtent(fTab.getLabel());

				e.gc.drawLine(textIndent, bounds.height - 4, textIndent + point.x, bounds.height - 4);
			}
		}

		/**
		 * Gets the text displayed on the tab.
		 * 
		 * @return string representing the text on the tab
		 */
		public String getText() {
			return fTab.getLabel();
		}

		@Override
		public String toString() {
			return fTab.getLabel();
		}
	}


	public class TopNavigationElement extends Canvas {

		/**
		 * @param parent
		 */
		public TopNavigationElement(Composite parent) {
			super(parent, SWT.NO_FOCUS);
			addPaintListener(new PaintListener() {

				public void paintControl(PaintEvent e) {
					paint(e);
				}
			});
			addMouseListener(new MouseAdapter() {

				public void mouseUp(MouseEvent e) {
					if (isUpScrollRequired()) {
						fBottomVisibleIndex--;
						if (fTopVisibleIndex != 0) {
							fTopVisibleIndex--;
						}
						layoutTabs();
						fTopNavigationElement.redraw();
						fBottomNavigationElement.redraw();
					}
				}
			});
		}

		/**
		 * @param e
		 */
		private void paint(PaintEvent e) {
			e.gc.setBackground(fDefaultBackground);
			e.gc.setForeground(fDefaultForeground);
			Rectangle bounds = getBounds();

			if (fElements.length != 0) {
				e.gc.fillRectangle(0, 0, bounds.width, bounds.height);
				e.gc.setForeground(fBorder);
				e.gc.drawLine(bounds.width - 1, 0, bounds.width - 1, bounds.height - 1);
			} else {
				e.gc.setBackground(fDefaultBackground);
				e.gc.fillRectangle(0, 0, bounds.width, bounds.height);
				int textIndent = INDENT;
				FontMetrics fm = e.gc.getFontMetrics();
				int height = fm.getHeight();
				int textMiddle = (bounds.height - height) / 2;

				e.gc.setForeground(fTextColor);
				e.gc.drawText(Messages2.TabbedList_properties_No_tabs_to_show, textIndent, textMiddle);
			}

			if (isUpScrollRequired()) {
				e.gc.setForeground(fDarkShadow);
				int middle = bounds.width / 2;

				e.gc.drawLine(middle + 1, 3, middle + 5, 7);
				e.gc.drawLine(middle, 3, middle - 4, 7);
				e.gc.drawLine(middle - 4, 8, middle + 5, 8);

				e.gc.setForeground(fActiveBackground);
				e.gc.drawLine(middle, 4, middle + 1, 4);
				e.gc.drawLine(middle - 1, 5, middle + 2, 5);
				e.gc.drawLine(middle - 2, 6, middle + 3, 6);
				e.gc.drawLine(middle - 3, 7, middle + 4, 7);
			}
		}
	}


	public class BottomNavigationElement extends Canvas {

		/**
		 * @param parent
		 */
		public BottomNavigationElement(Composite parent) {
			super(parent, SWT.NO_FOCUS);
			addPaintListener(new PaintListener() {

				public void paintControl(PaintEvent e) {
					paint(e);
				}
			});
			addMouseListener(new MouseAdapter() {

				public void mouseUp(MouseEvent e) {
					if (isDownScrollRequired()) {
						fTopVisibleIndex++;
						if (fBottomVisibleIndex != fElements.length - 1) {
							fBottomVisibleIndex++;
						}
						layoutTabs();
						fTopNavigationElement.redraw();
						fBottomNavigationElement.redraw();
					}
				}
			});
		}

		/**
		 * @param e
		 */
		private void paint(PaintEvent e) {
			e.gc.setBackground(fDefaultBackground);
			e.gc.setForeground(fDefaultForeground);
			Rectangle bounds = getBounds();

			if (fElements.length != 0) {
				e.gc.fillRectangle(0, 0, bounds.width, bounds.height);
				e.gc.setForeground(fBorder);
				e.gc.drawLine(bounds.width - 1, 0, bounds.width - 1, bounds.height - 1);
				if (getSelectionIndex() >= 0 && fElements.length != 0 && getSelectionIndex() == fElements.length - 1) {
					e.gc.drawLine(4, 0, bounds.width - 1, 0);
				} else {
					e.gc.drawLine(2, 0, bounds.width - 1, 0);
				}
			} else {
				e.gc.setBackground(fDefaultBackground);
				e.gc.fillRectangle(0, 0, bounds.width, bounds.height);
			}

			if (isDownScrollRequired()) {
				e.gc.setForeground(fDarkShadow);
				int middle = bounds.width / 2;
				int bottom = bounds.height - 2;

				e.gc.drawLine(middle + 1, bottom, middle + 5, bottom - 4);
				e.gc.drawLine(middle, bottom, middle - 4, bottom - 4);
				e.gc.drawLine(middle - 4, bottom - 5, middle + 5, bottom - 5);

				e.gc.setForeground(fActiveBackground);
				e.gc.drawLine(middle, bottom - 1, middle + 1, bottom - 1);
				e.gc.drawLine(middle - 1, bottom - 2, middle + 2, bottom - 2);
				e.gc.drawLine(middle - 2, bottom - 3, middle + 3, bottom - 3);
				e.gc.drawLine(middle - 3, bottom - 4, middle + 4, bottom - 4);
			}
		}
	}

	/**
	 * Ctor.
	 * 
	 * @param parent
	 *            the parent composite widget
	 * @param factory
	 *            the widget factory used to create the widgets
	 */
	public TabbedListEpl(Composite parent, TabbedWidgetFactoryEpl factory) {
		super(parent, SWT.NO_FOCUS);
		this.fFactory = factory;
		removeAll();
		setLayout(new FormLayout());
		initColours();
		initAccessible();
		fTopNavigationElement = new TopNavigationElement(this);
		fBottomNavigationElement = new BottomNavigationElement(this);

		this.addFocusListener(new FocusListener() {

			public void focusGained(FocusEvent e) {
				fFocus = true;
				int i = getSelectionIndex();

				if (i >= 0) {
					fElements[i].redraw();
				}
			}

			public void focusLost(FocusEvent e) {
				fFocus = false;
				int i = getSelectionIndex();

				if (i >= 0) {
					fElements[i].redraw();
				}
			}
		});
		this.addControlListener(new ControlAdapter() {

			public void controlResized(ControlEvent e) {
				computeTopAndBottomTab();
			}
		});
		this.addTraverseListener(new TraverseListener() {

			public void keyTraversed(TraverseEvent e) {
				if (e.detail == SWT.TRAVERSE_ARROW_PREVIOUS || e.detail == SWT.TRAVERSE_ARROW_NEXT) {
					int nMax = fElements.length - 1;
					int nCurrent = getSelectionIndex();

					if (e.detail == SWT.TRAVERSE_ARROW_PREVIOUS) {
						nCurrent -= 1;
						nCurrent = Math.max(0, nCurrent);
					} else if (e.detail == SWT.TRAVERSE_ARROW_NEXT) {
						nCurrent += 1;
						nCurrent = Math.min(nCurrent, nMax);
					}
					select(nCurrent, true);
					redraw();
				} else {
					e.doit = true;
				}
			}
		});
	}

	/**
	 * Calculate the number of tabs that will fit in the tab list composite.
	 */
	protected void computeTabsThatFitInComposite() {
		float size = getSize().y - 22;
		float tabHeight = getTabHeight();
		fTabsThatFitInComposite = Math.round(size / tabHeight);
		if (fTabsThatFitInComposite <= 0) {
			fTabsThatFitInComposite = 1;
		}
	}

	/**
	 * Returns the element with the given index from this list viewer. Returns
	 * <code>null</code> if the index is out of range.
	 * 
	 * @param index
	 *            the zero-based index
	 * @return the element at the given index, or <code>null</code> if the
	 *         index is out of range
	 */
	public Object getElementAt(int index) {
		if (index >= 0 && index < fElements.length) {
			return fElements[index];
		}
		return null;
	}

	/**
	 * Returns the zero-relative index of the item which is currently selected
	 * in the receiver, or -1 if no item is selected.
	 * 
	 * @return the index of the selected item
	 */
	public int getSelectionIndex() {
		return fSelectedElementIndex;
	}

	/**
	 * Removes all elements from this list.
	 */
	public void removeAll() {
		if (fElements != null) {
			for (int i = 0; i < fElements.length; i++) {
				fElements[i].dispose();
			}
		}
		fElements = ELEMENTS_EMPTY;
		fSelectedElementIndex = -1;
		fWidestLabelIndex = -1;
		fTopVisibleIndex = -1;
		fBottomVisibleIndex = -1;
	}

	/**
	 * Sets the new list elements.
	 */
	public void setElements(ITabDescriptorEpl[] children) {
		if (fElements != ELEMENTS_EMPTY) {
			removeAll();
		}
		fElements = new ListElement[children.length];
		if (children.length == 0) {
			fWidestLabelIndex = -1;
		} else {
			fWidestLabelIndex = 0;
			for (int i = 0; i < children.length; i++) {
				fElements[i] = new ListElement(this, children[i], i);
				fElements[i].setVisible(false);
				fElements[i].setLayoutData(null);

				if (i != fWidestLabelIndex) {
					String label = (children[i]).getLabel();

					if (getTextDimension(label).x
							> getTextDimension((children[fWidestLabelIndex]).getLabel()).x) {
						fWidestLabelIndex = i;
					}
				}
			}
		}

		computeTopAndBottomTab();
	}

	/**
	 * Selects one of the elements in the list.
	 * 
	 * @param index
	 *            the list index
	 * @param reveal
	 *            not used
	 */
	public void select(int index, boolean reveal) {
		if (fSelectedElementIndex == index) {
			return; // already selected
		}
		if (index >= 0 && index < fElements.length) {
			int lastSelected = fSelectedElementIndex;

			fElements[index].setSelected(true);
			fSelectedElementIndex = index;
			if (lastSelected >= 0) {
				fElements[lastSelected].setSelected(false);
				if (fSelectedElementIndex != fElements.length - 1) {

					/*
					 * redraw the next tab to fix the border by calling
					 * setSelected()
					 */
					fElements[fSelectedElementIndex + 1].setSelected(false);
				}
			}
			fTopNavigationElement.redraw();
			fBottomNavigationElement.redraw();

			if (fSelectedElementIndex < fTopVisibleIndex || fSelectedElementIndex > fBottomVisibleIndex) {
				computeTopAndBottomTab();
			}
		}
		notifyListeners(SWT.Selection, new Event());
	}

	/**
	 * Deselects all the elements in the list.
	 */
	public void deselectAll() {
		if (getSelectionIndex() >= 0) {
			fElements[getSelectionIndex()].setSelected(false);
			fSelectedElementIndex = -1;
		}
	}

	/**
	 * Computes the size based on the widest string in the list.
	 */
	@Override
	public Point computeSize(int wHint, int hHint, boolean changed) {
		Point result = super.computeSize(hHint, wHint, changed);

		if (fWidestLabelIndex < 0) {
			result.x = getTextDimension(Messages2.TabbedList_properties_No_tabs_to_show).x + INDENT;
		} else {
			int width = getTextDimension(fElements[fWidestLabelIndex].getText()).x;

			/*
			 * To anticipate for the icon placement we should always keep the
			 * space available after the label. So when the active tab includes
			 * an icon the width of the tab doesn't change.
			 */
			result.x = width + ANTICIPATE_ICON;
		}
		return result;
	}

	/**
	 * Proper cleanup of the widgets.
	 * 
	 * @see org.eclipse.swt.widgets.Widget#dispose()
	 */
	@Override
	public void dispose() {
		fHoverBackground.dispose();
		fDefaultBackground.dispose();
		fDefaultForeground.dispose();
		fActiveBackground.dispose();
		fBorder.dispose();
		fDarkShadow.dispose();
		fTextColor.dispose();
		super.dispose();
	}

	/**
	 * Get the index for a certain element in the list.
	 * 
	 * @param element
	 *            the list element
	 * @return the index in the list of the given element
	 */
	private int getIndex(ListElement element) {
		return element.fIndex;
	}

	/**
	 * Get the text dimension as a point.
	 * 
	 * @param text
	 *            the string
	 * @return a point containign the text dimensions
	 */
	private Point getTextDimension(String text) {
		Shell shell = new Shell();
		GC gc = new GC(shell);

		gc.setFont(getFont());
		Point point = gc.textExtent(text);

		point.x++;
		gc.dispose();
		shell.dispose();
		return point;
	}

	/**
	 * Initialize the colours used.
	 */
	private void initColours() {
		fDefaultBackground = Display.getCurrent().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND);

		RGB rgb = fDefaultBackground.getRGB();

		rgb.blue = Math.min(255, Math.round(rgb.blue * 1.05F));
		rgb.red = Math.min(255, Math.round(rgb.red * 1.05F));
		rgb.green = Math.min(255, Math.round(rgb.green * 1.05F));
		fHoverBackground = fFactory.getColors().createColor("TabbedListEpl.hoverBackground", rgb); //$NON-NLS-1$

		fDefaultForeground = Display.getCurrent().getSystemColor(SWT.COLOR_WIDGET_FOREGROUND);
		fActiveBackground = Display.getCurrent().getSystemColor(SWT.COLOR_LIST_BACKGROUND);
		fBorder = Display.getCurrent().getSystemColor(SWT.COLOR_WIDGET_NORMAL_SHADOW);
		fDarkShadow = Display.getCurrent().getSystemColor(SWT.COLOR_WIDGET_DARK_SHADOW);
		fTextColor = Display.getCurrent().getSystemColor(SWT.COLOR_WIDGET_FOREGROUND);
	}

	/**
	 * Get the height of a tab. The height of the tab is the height of the text
	 * plus buffer.
	 * 
	 * @return the height of a tab.
	 */
	private int getTabHeight() {
		int tabHeight = getTextDimension("").y + INDENT; //$NON-NLS-1$

		if (fTabsThatFitInComposite == 1) {

			/*
			 * if only one tab will fix, reduce the size of the tab height so
			 * that the navigation elements fit.
			 */
			int ret = getBounds().height - 20;

			return (ret > tabHeight) ? tabHeight : (ret < 5) ? 5 : ret;
		}
		return tabHeight;
	}

	/**
	 * Check to see if scrolling down the list of tabs is required.
	 * 
	 * @return <code>true</code> if the list needs scrolling dowwn or
	 *         <code>false</code> otherwise
	 */
	private boolean isDownScrollRequired() {
		return fElements.length > fTabsThatFitInComposite && fBottomVisibleIndex != fElements.length - 1;
	}

	/**
	 * Check to see if scrolling up the list of tabs is required.
	 * 
	 * @return <code>true</code> if the list needs scrolling up or
	 *         <code>false</code> otherwise
	 */
	private boolean isUpScrollRequired() {
		return fElements.length > fTabsThatFitInComposite && fTopVisibleIndex != 0;
	}

	/**
	 * The tabs might not fit all into the composite and when 
	 * one is selected, based on its position, the other tabs 
	 * might need repositioning. Calls <code>layoutTabs</code> 
	 * to layout the tabs. 
	 */
	private void computeTopAndBottomTab() {
		computeTabsThatFitInComposite();
		if (fElements.length == 0) {

			/*
			 * no tabs to display.
			 */
			fTopVisibleIndex = 0;
			fBottomVisibleIndex = 0;
		} else if (fTabsThatFitInComposite >= fElements.length) {

			/*
			 * all the tabs fit.
			 */
			fTopVisibleIndex = 0;
			fBottomVisibleIndex = fElements.length - 1;
		} else if (getSelectionIndex() < 0) {

			/*
			 * there is no selected tab yet, assume that tab one would be
			 * selected for now.
			 */
			fTopVisibleIndex = 0;
			fBottomVisibleIndex = fTabsThatFitInComposite - 1;
		} else if (getSelectionIndex() + fTabsThatFitInComposite > fElements.length) {

			/*
			 * the selected tab is near the bottom.
			 */
			fBottomVisibleIndex = fElements.length - 1;
			fTopVisibleIndex = fBottomVisibleIndex - fTabsThatFitInComposite + 1;
		} else {

			/*
			 * the selected tab is near the top.
			 */
			fTopVisibleIndex = fSelectedElementIndex;
			fBottomVisibleIndex = fSelectedElementIndex + fTabsThatFitInComposite - 1;
		}
		layoutTabs();
	}

	/**
	 * Layout the tabs.
	 * 
	 * @param up
	 *            if <code>true</code>, then we are laying out as a result of
	 *            a scroll up request.
	 */
	private void layoutTabs() {
		// System.out.println("TabFit " + tabsThatFitInComposite + " length "
		// + elements.length + " top " + topVisibleIndex + " bottom "
		// + bottomVisibleIndex);
		if (fTabsThatFitInComposite < 0 || fElements.length == 0) {
			FormData formData = new FormData();

			formData.left = new FormAttachment(0, 0);
			formData.right = new FormAttachment(100, 0);
			formData.top = new FormAttachment(0, 0);
			formData.height = getTabHeight();
			fTopNavigationElement.setLayoutData(formData);

			formData = new FormData();
			formData.left = new FormAttachment(0, 0);
			formData.right = new FormAttachment(100, 0);
			formData.top = new FormAttachment(fTopNavigationElement, 0);
			formData.bottom = new FormAttachment(100, 0);
			fBottomNavigationElement.setLayoutData(formData);
		} else {

			FormData formData = new FormData();

			formData.left = new FormAttachment(0, 0);
			formData.right = new FormAttachment(100, 0);
			formData.top = new FormAttachment(0, 0);
			formData.height = 10;
			fTopNavigationElement.setLayoutData(formData);

			/*
			 * use nextElement to attach the layout to the previous canvas
			 * widget in the list.
			 */
			Canvas nextElement = fTopNavigationElement;

			for (int i = 0; i < fElements.length; i++) {
				// System.out.print(i + " [" + elements[i].getText() + "]");
				if (i < fTopVisibleIndex || i > fBottomVisibleIndex) {

					/*
					 * this tab is not visible
					 */
					fElements[i].setLayoutData(null);
					fElements[i].setVisible(false);
				} else {

					/*
					 * this tab is visible.
					 */
					// System.out.print(" visible");
					formData = new FormData();
					formData.height = getTabHeight();
					formData.left = new FormAttachment(0, 0);
					formData.right = new FormAttachment(100, 0);
					formData.top = new FormAttachment(nextElement, 0);
					nextElement = fElements[i];
					fElements[i].setLayoutData(formData);
					fElements[i].setVisible(true);
				}

				// if (i == selectedElementIndex) {
				// System.out.print(" selected");
				// }
				// System.out.println("");
			}
			formData = new FormData();
			formData.left = new FormAttachment(0, 0);
			formData.right = new FormAttachment(100, 0);
			formData.top = new FormAttachment(nextElement, 0);
			formData.bottom = new FormAttachment(100, 0);
			formData.height = 10;
			fBottomNavigationElement.setLayoutData(formData);
		}
		// System.out.println("");

		// layout so that we have enough space for the new labels
		Composite grandparent = getParent().getParent();

		grandparent.layout(true);
		layout(true);
	}

	/**
	 * Initialize the accessibility adapter.
	 */
	private void initAccessible() {
		final Accessible accessible = getAccessible();

		accessible.addAccessibleListener(new AccessibleAdapter() {
            @Override
			public void getName(AccessibleEvent e) {
				if (getSelectionIndex() >= 0) {
					e.result = fElements[getSelectionIndex()].getText();
				}
			}
            @Override
			public void getHelp(AccessibleEvent e) {
				if (getSelectionIndex() >= 0) {
					e.result = fElements[getSelectionIndex()].getText();
				}
			}
		});

		accessible.addAccessibleControlListener(
				new AccessibleControlAdapter() {
			@Override
			public void getChildAtPoint(AccessibleControlEvent e) {
				Point pt = toControl(new Point(e.x, e.y));

				e.childID = (getBounds().contains(pt)) ? ACC.CHILDID_SELF : ACC.CHILDID_NONE;
			}
			@Override
			public void getLocation(AccessibleControlEvent e) {
				if (getSelectionIndex() >= 0) {
					Rectangle location = fElements[getSelectionIndex()].getBounds();
					Point pt = toDisplay(new Point(location.x, location.y));

					e.x = pt.x;
					e.y = pt.y;
					e.width = location.width;
					e.height = location.height;
				}
			}
			@Override
			public void getChildCount(AccessibleControlEvent e) {
				e.detail = 0;
			}
			@Override 
			public void getRole(AccessibleControlEvent e) {
				e.detail = ACC.ROLE_TABITEM;
			}
			@Override
			public void getState(AccessibleControlEvent e) {
				e.detail = ACC.STATE_NORMAL | ACC.STATE_SELECTABLE | ACC.STATE_SELECTED | ACC.STATE_FOCUSED
						| ACC.STATE_FOCUSABLE;
			}
		});

		addListener(SWT.Selection, new Listener() {
			@Override
			public void handleEvent(Event event) {
				if (isFocusControl()) {
					accessible.setFocus(ACC.CHILDID_SELF);
				}
			}
		});

		addListener(SWT.FocusIn, new Listener() {
			@Override
			public void handleEvent(Event event) {
				accessible.setFocus(ACC.CHILDID_SELF);
			}
		});
	}
}
