/*******************************************************************************
 * Copyright 2023 NXP
 * SPDX-License-Identifier: EPL-1.0
 *******************************************************************************/

package com.freescale.system.browser.epl.help;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.ui.plugin.AbstractUIPlugin;

import com.freescale.system.browser.epl.ISectionDescriptorEpl;
import com.freescale.system.browser.epl.TabDescriptorEpl;
import com.freescale.system.browser.epl.TabEpl;
import com.freescale.system.browser2.Messages2;
import com.freescale.system.browser2.SystemBrowserActivator2;
import com.freescale.system.browser2.ISimpleTableDataProvider2;
import com.freescale.system.browser.epl.ISectionEpl;


public class TasksTabDescriptorEpl extends TabDescriptorEpl {
	// XML elements and attributes used to declare the desire of and
	// customization of the built in Tasks tab.
	private static final String ELEM_COLUMN = "systemTabTasksColumn"; //$NON-NLS-1$
	private static final String ATT_TILE = "title"; //$NON-NLS-1$
	private static final String ATT_ICON = "icon"; //$NON-NLS-1$
	private static final String ATT_DATAID = "dataID"; //$NON-NLS-1$
	private static final String ATT_WIDTH = "width"; //$NON-NLS-1$
	private static final String ATT_IS_NUMERIC = "isNumeric"; //$NON-NLS-1$
	private static final String ATT_COLUMN_LABEL = "label"; //$NON-NLS-1$
	private static final String ATT_NAME_COL_SHOW = "columnNameShow"; //$NON-NLS-1$
	private static final String ATT_NAME_COL_WIDTH = "columnNameWidth"; //$NON-NLS-1$
	private static final String ATT_STATUS_COL_SHOW = "columnStatusShow"; //$NON-NLS-1$
	private static final String ATT_ID_COL_SHOW = "columnIDShow"; //$NON-NLS-1$
	private static final String ATT_ID_COL_HEX = "columnIDHex"; //$NON-NLS-1$
	//private static final String ATT_SCOPE_IS_MACHINE = "scopeIsMachine"; //$NON-NLS-1$	
	protected static final String BUILTIN_COLUMN_DATAID_NAME = "name"; //$NON-NLS-1$ 
	protected static final String BUILTIN_COLUMN_DATAID_ID = "id"; //$NON-NLS-1$
	protected static final String BUILTIN_COLUMN_DATAID_STATUS = "status"; //$NON-NLS-1$
	
	/**
	 * Structure that describes a column in the Tasks table. Used for
	 * both built-in columns and custom ones. 
	 */
	public static class Column {

		/**
		 * Use this for custom columns
		 */
		public Column(String label, String dataId, int width, boolean isNumeric) {
			this.label = label;
			this.dataId = dataId;
			this.width = width;
			this.isNumeric = isNumeric;
			this.builtIn = false;
		}
		
		/**
		 * Use this for built-in columns
		 */
		public Column(String label, String dataId, int width) {
			this.label = label;
			this.dataId = dataId;
			this.width = width;
			this.builtIn = true;
			this.isNumeric = false;
		}
		
		public final String label;
		public final String dataId;
		public final int width;
		public final boolean builtIn;
		public final boolean isNumeric; // n/a if builtIn == true
	}
	
	/**
	 * Collection of custom columns declared in the extension XML
	 */
	private final List<Column> fCustomColumns = new LinkedList<>();
	//private fTaskExtendedDecriptor;

	// Optional attributes; stored here in string form.
	private String fShowNameColumn;
	private String fWidthNameColumn;
	private String fShowStateColumn;
	private String fShowIdColumn;
	private String fHexIdColumn;
	//private String fScopeIsMachine;
	private ISimpleTableDataProvider2 fTaskData;
	
	/**
	 * Constructor
	 * 
	 * @param configurationElement
	 *            the systemTasksTab element (XML) of the systemTabs extension.
	 */
	public TasksTabDescriptorEpl(IConfigurationElement configurationElement) {
		super(configurationElement);
		
		fTitle = configurationElement.getAttribute(ATT_TILE);
		if (fTitle == null) {
			fTitle = Messages2.TasksTab_Default_Label;
		}

		String iconstr = configurationElement.getAttribute(ATT_ICON);
		if (iconstr != null) {
			fIcon = AbstractUIPlugin.imageDescriptorFromPlugin(
					configurationElement.getDeclaringExtension().getNamespaceIdentifier(), iconstr);
		}

		fShowNameColumn = configurationElement.getAttribute(ATT_NAME_COL_SHOW);
		fWidthNameColumn = configurationElement.getAttribute(ATT_NAME_COL_WIDTH);
		fShowStateColumn = configurationElement.getAttribute(ATT_STATUS_COL_SHOW);
		fShowIdColumn = configurationElement.getAttribute(ATT_ID_COL_SHOW);		
		fHexIdColumn = configurationElement.getAttribute(ATT_ID_COL_HEX);
		//fScopeIsMachine = configurationElement.getAttribute(ATT_SCOPE_IS_MACHINE);

		IConfigurationElement[] children = configurationElement.getChildren(ELEM_COLUMN);
		for (IConfigurationElement child : children) {
			final String label = child.getAttribute(ATT_COLUMN_LABEL);
			final String dataId = child.getAttribute(ATT_DATAID);
			int width = 200;
			boolean isNumeric = false;			
			
			String value = child.getAttribute(ATT_WIDTH);
			if (value != null) {
				try {
					width = Integer.parseInt(value);
				} catch (NumberFormatException exc) {
					SystemBrowserActivator2.log(IStatus.ERROR,
							"System Browser: invalid custom column width specification"); //$NON-NLS-1$ 
				}
			}

			value = child.getAttribute(ATT_IS_NUMERIC);
			if (value != null) {
				try {
					isNumeric = Boolean.parseBoolean(value);
				} catch (NumberFormatException exc) {
					SystemBrowserActivator2.log(IStatus.ERROR,
							"System Browser: invalid custom column isNumeric specification"); //$NON-NLS-1$ 
				}
			}
			
			if (label != null && dataId != null) { 
				fCustomColumns.add(new Column(label, dataId, width, isNumeric));
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.TabDescriptor#createTab()
	 */
	@Override
	protected TabEpl createTabEx() {
		
		// Set up the Options struct based on what we found in the extension XML
		
		TasksSectionEpl.Options options = new TasksSectionEpl.Options(); 
		if (fShowNameColumn != null) {
			options.hideNameColumn = !Boolean.parseBoolean(fShowNameColumn);
		}

		if (fShowStateColumn != null) {
			options.hideStatusColumn = !Boolean.parseBoolean(fShowStateColumn);
		}
		if (fWidthNameColumn != null) {
			try {
				options.nameColumnWidth = Integer.parseInt(fWidthNameColumn);
			} catch (NumberFormatException exc) {}
		}
		
		if (fShowIdColumn != null) {
			options.hideIdColumn = !Boolean.parseBoolean(fShowIdColumn);
		}
		
		if (fHexIdColumn != null) {
			options.hexIdColumn = Boolean.parseBoolean(fHexIdColumn);
		}
		if (fTaskData != null) {
			options.taskData =  fTaskData;
		}

		// Create the section
		TasksSectionEpl tasksSection = new TasksSectionEpl(//
				fCustomColumns.toArray(new Column[fCustomColumns.size()]), //
				options);
		List<ISectionEpl> taskList= new ArrayList<>();
		taskList.add(tasksSection);
		// create the tab instance
		TabEpl tab = new TabEpl(this, taskList);
		return tab;
	}

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.TabDescriptor#getId()
	 */
	@Override
	public String getId() {
		// the IDs of a built in tab used only once to .
		return getClass().toString();
	}

	@Override
	protected void addSectionDescriptor(final ISectionDescriptorEpl section) {
		if (section instanceof SimpleTableDataDescriptorEpl) {
			fTaskData = ((SimpleTableDataDescriptorEpl)section).createDataProvider();
		}
	}
	
	/** */
	public final ISimpleTableDataProvider2 getTableDataObject() {
		return fTaskData;
	}

}
