/**
* Copyright 2019-2020, 2023 NXP
*
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*/ 

package com.nxp.s32ds.ext.rcp.utils.epl.viewers;

import org.eclipse.core.runtime.ListenerList;
import org.eclipse.jface.util.SafeRunnable;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.ITreeViewerListener;
import org.eclipse.jface.viewers.TreeExpansionEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Item;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.swt.widgets.Widget;

/**
 * It is analog of {@link org.eclipse.ui.dialogs.ContainerCheckedTreeViewer} but
 * adapted to work with {@link org.eclipse.ui.dialogs.FilteredTree} component.
 *
 */
public class ContainerFilteredCheckboxTreeViewer extends CheckboxTreeViewer {

	private final TreeContentCheckStateHolder stateHolder = new TreeContentCheckStateHolder(() -> getContentProvider());

	private final TreeContentCheckStateProvider stateProvider = new TreeContentCheckStateProvider(stateHolder);

	public interface StrucureChangedListener {
		void structureChanged(TreeViewer treeViewer);
	}

	private final ListenerList<StrucureChangedListener> listenerList = new ListenerList<>();

	private boolean insidePreservingSelection;

	public ContainerFilteredCheckboxTreeViewer(Composite parent) {
		super(parent);
		initViewer();

	}

	public ContainerFilteredCheckboxTreeViewer(Composite parent, int style) {
		super(parent, style);
		initViewer();
	}

	public ContainerFilteredCheckboxTreeViewer(Tree tree) {
		super(tree);
		initViewer();
	}

	@Override
	public void setAllChecked(boolean state) {
		ITreeContentProvider contentProvider = getContentProvider();
		if (contentProvider == null) {
			throw new IllegalStateException("setAllChecked method should be called after setContentProvider"); //$NON-NLS-1$
		}
		for (Object object : contentProvider.getElements(getInput())) {
			stateHolder.setChecked(object, state);
		}
		refresh();
	}

	private void initViewer() {
		setCheckStateProvider(stateProvider);
		setUseHashlookup(true);
		addCheckStateListener(new ICheckStateListener() {
			@Override
			public void checkStateChanged(CheckStateChangedEvent event) {
				doCheckStateChanged(event.getElement(), event.getChecked());
			}
		});

		addTreeListener(new ITreeViewerListener() {
			@Override
			public void treeCollapsed(TreeExpansionEvent event) {
				// do nothing
			}

			@Override
			public void treeExpanded(TreeExpansionEvent event) {
				initializeElement(event.getElement());
			}

		});
	}

	@Override
	public Object[] getCheckedElements() {
		return stateHolder.getCheckedElements();
	}

	@Override
	public void setCheckedElements(Object[] elements) {
		super.setCheckedElements(elements);
		for (int i = 0; i < elements.length; i++) {
			doCheckStateChanged(elements[i], true);
		}
	}

	@Override
	protected void setExpanded(Item item, boolean expand) {
		super.setExpanded(item, expand);
		if (expand && item instanceof TreeItem) {
			initializeElement(item.getData());
		}
	}

	protected void doCheckStateChanged(Object element, boolean checked) {
		if (!insidePreservingSelection) {
			stateHolder.setChecked(element, checked);
		}
		TreeItem treeItem = findTreeItem(element);
		if (treeItem == null) {
			return;
		}
		treeItem.setGrayed(false);
		updateChildrenItems(treeItem);
		updateParentItems(treeItem.getParentItem());
	}

	protected TreeItem findTreeItem(Object element) {
		Widget widget = super.doFindItem(element);
		if (widget instanceof TreeItem) {
			return (TreeItem) widget;
		}
		return null;
	}

	protected void initializeElement(Object object) {
		TreeItem treeItem = findTreeItem(object);
		if (treeItem == null) {
			return;
		}
		updateChildrenItems(treeItem);
	}

	protected void updateChildrenItems(TreeItem treeItem) {
		Item[] children = getChildren(treeItem);
		for (int i = 0; i < children.length; i++) {
			TreeItem curr = (TreeItem) children[i];
			if (updateItemCheckedState(curr)) {
				updateChildrenItems(curr);
			}
		}
	}

	protected void updateParentItems(TreeItem treeItem) {
		if (treeItem == null) {
			return;
		}
		if (updateItemCheckedState(treeItem)) {
			updateParentItems(treeItem.getParentItem());
		}
	}

	private boolean updateItemCheckedState(TreeItem treeItem) {
		Object data = treeItem.getData();
		if (data == null) {
			return false;
		}
		treeItem.setGrayed(stateProvider.isGrayed(data));
		treeItem.setChecked(stateProvider.isChecked(data));
		return true;
	}

	protected boolean isInsidePreservingSelection() {
		return insidePreservingSelection;
	}

	@Override
	protected void preservingSelection(Runnable updateCode) {
		insidePreservingSelection = true;
		try {
			super.preservingSelection(updateCode);
		} finally {
			insidePreservingSelection = false;
		}
	}

	@Override
	public ITreeContentProvider getContentProvider() {
		// setContentProvider check for instance of ITreeContentProvider
		return (ITreeContentProvider) super.getContentProvider();
	}

	// Structure changed notification - start
	public void addStrucureChangedListener(StrucureChangedListener listener) {
		listenerList.add(listener);
	}

	public void removeStrucureChangedListener(StrucureChangedListener listener) {
		listenerList.remove(listener);
	}

	protected void fireStrucureChanged() {
		for (StrucureChangedListener listener : listenerList) {
			SafeRunnable.run(new SafeRunnable() {
				@Override
				public void run() {
					listener.structureChanged(ContainerFilteredCheckboxTreeViewer.this);
				}
			});
		}
	}

	@Override
	public void add(Object parentElementOrTreePath, Object childElement) {
		fireStrucureChanged();
		super.add(parentElementOrTreePath, childElement);
	}

	@Override
	public void add(Object parentElementOrTreePath, Object... childElements) {
		fireStrucureChanged();
		super.add(parentElementOrTreePath, childElements);
	}

	@Override
	protected void inputChanged(Object input, Object oldInput) {
		stateHolder.clear();
		fireStrucureChanged();
		super.inputChanged(input, oldInput);
	}

	@Override
	public void insert(Object parentElementOrTreePath, Object element, int position) {
		fireStrucureChanged();
		super.insert(parentElementOrTreePath, element, position);
	}

	@Override
	public void refresh() {
		fireStrucureChanged();
		super.refresh();
	}

	@Override
	public void refresh(boolean updateLabels) {
		fireStrucureChanged();
		super.refresh(updateLabels);
	}

	@Override
	public void refresh(Object element) {
		fireStrucureChanged();
		super.refresh(element);
	}

	@Override
	public void refresh(Object element, boolean updateLabels) {
		fireStrucureChanged();
		super.refresh(element, updateLabels);
	}

	@Override
	public void remove(Object elementsOrTreePaths) {
		fireStrucureChanged();
		super.remove(elementsOrTreePaths);
	}

	@Override
	public void remove(Object parent, Object... elements) {
		fireStrucureChanged();
		super.remove(parent, elements);
	}

	@Override
	public void remove(Object... elementsOrTreePaths) {
		fireStrucureChanged();
		super.remove(elementsOrTreePaths);
	}

	@Override
	public void replace(Object parentElementOrTreePath, int index, Object element) {
		fireStrucureChanged();
		super.replace(parentElementOrTreePath, index, element);
	}

	@Override
	public void setChildCount(Object elementOrTreePath, int count) {
		fireStrucureChanged();
		super.setChildCount(elementOrTreePath, count);
	}

	@Override
	public void setContentProvider(IContentProvider provider) {
		if (!(provider instanceof ITreeContentProvider)) {
			throw new IllegalArgumentException("provider smust be instance of ITreeContentProvider"); //$NON-NLS-1$
		}
		fireStrucureChanged();
		super.setContentProvider(provider);
	}

	@Override
	public void setHasChildren(Object elementOrTreePath, boolean hasChildren) {
		fireStrucureChanged();
		super.setHasChildren(elementOrTreePath, hasChildren);
	}
	// Structure changed notification - end
}