/**
 * Copyright 2020 NXP
 * Created: 27 Aug 2020
 */
package com.nxp.swtools.periphs.gui.dialogs.initializationorder;

import java.util.LinkedList;
import java.util.List;

import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.periphs.model.data.InitializationPrioritiesGroup;
import com.nxp.swtools.resourcetables.model.data.Description;

/**
 * Wrapper containing the initialization priority group
 * @author Tomas Rudolf - nxf31690
 */
public class InitializationOrderGroup {
	/** Initialization priority groups */
	private final InitializationPrioritiesGroup group;
	/** List of instance wrappers */
	private LinkedList<InitializationOrderInstance> instances = new LinkedList<>();

	/**
	 * Constructor
	 * @param group the initialization priority group
	 */
	public InitializationOrderGroup(InitializationPrioritiesGroup group) {
		this.group = group;
	}

	/**
	 * @return the instance wrappers
	 */
	public List<InitializationOrderInstance> getInstances() {
		return instances;
	}

	/**
	 * Sets the instance wrappers
	 * @param instances instance wrappers
	 */
	public void setInstances(List<InitializationOrderInstance> instances) {
		this.instances = new LinkedList<>(instances);
	}

	/**
	 * Returns the name of the group which is label if it was specified, otherwise returns id
	 * @return the name of the group
	 */
	public String getName() {
		String label = group.getLabel();
		return label == null ? group.getId() : label;
	}

	/**
	 * @return the description of the group
	 */
	public @Nullable String getDescription() {
		Description description = group.getDescription();
		if (description == null) {
			return null;
		}
		return description.getDescriptionText();
	}

	/**
	 * Checks whether given instance can be moved by one position before its current position
	 * @param instance to be moved
	 * @return {@code true} when instance can be moved, {@code false} otherwise
	 */
	public boolean canMoveBefore(InitializationOrderInstance instance) {
		int index = instances.indexOf(instance);
		if (index == -1) {
			return false;
		}
		if (index - 1 < 0) {
			return false; // Cannot move the first one
		}
		return true;
	}

	/**
	 * Tries to move the given instance by one position before its current position
	 * @param instance to be moved
	 * @return {@code true} when instance was moved, {@code false} otherwise
	 */
	public boolean moveBefore(InitializationOrderInstance instance) {
		if (canMoveBefore(instance)) {
			int index = instances.indexOf(instance);
			instances.add(index - 1, instance);
			instances.remove(index + 1);
			return true;
		}
		return false;
	}

	/**
	 * Checks whether given instance can be moved by one position after its current position
	 * @param instance to be moved
	 * @return {@code true} when instance can be moved, {@code false} otherwise
	 */
	public boolean canMoveAfter(InitializationOrderInstance instance) {
		int index = instances.indexOf(instance);
		if (index == -1) {
			return false;
		}
		if (index + 1 >= instances.size()) {
			return false; // Cannot move the last one
		}
		return true;
	}

	/**
	 * Tries to move the given instance by one position after its current position
	 * @param instance to be moved
	 * @return {@code true} when instance was moved, {@code false} otherwise
	 */
	public boolean moveAfter(InitializationOrderInstance instance) {
		if (canMoveAfter(instance)) {
			int index = instances.indexOf(instance);
			instances.add(index + 2, instance);
			instances.remove(index);
			return true;
		}
		return false;
	}
}