/**
 * Copyright 2021-2022 NXP
 * Created: Mar 22 2021
 */
package com.nxp.swtools.periphs.gui.templates.component;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.ui.dialogs.FilteredTree;
import org.eclipse.ui.dialogs.PatternFilter;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.view.AddComponentDialog;
import com.nxp.swtools.periphs.model.data.AvailableComponents;
import com.nxp.swtools.periphs.model.templates.component.ComponentTemplate;
import com.nxp.swtools.periphs.model.templates.component.ComponentTemplateProperties;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsImages;

/**
 * Dialog for managing use cases in the use case library
 * @author Tomas Rudolf nxf31690
 */
public class ComponentTemplateManagerDialog extends Dialog {

	/**
	 * Entry of the tree viewer used in {@link ComponentTemplateManagerDialog}
	 * @author Tomas Rudolf - nxf31690
	 */
	private static class TreeEntry {
		/** Type of component which this entry is related to */
		private String type;
		/** TypeId of component which this entry is related to */
		private String typeId;
		/** List of sub entries if this is component entry */
		private List<TreeEntry> subEntries = Collections.emptyList();
		/** Template properties if this is template entry */
		private @Nullable ComponentTemplateProperties properties;
		/** Parent entry */
		private @Nullable TreeEntry parent;
		/** Flag whether entry is checked in viewer */
		private boolean isChecked;

		/**
		 * Constructor for component entry
		 * @param type of the component
		 * @param typeId of the component
		 */
		public TreeEntry(String type, String typeId) {
			this.type = type;
			this.typeId = typeId;
		}

		/**
		 * Constructor for the template entry
		 * @param parent component entry
		 * @param properties of the template
		 */
		public TreeEntry(TreeEntry parent, ComponentTemplateProperties properties) {
			this.type = parent.getType();
			this.typeId = parent.getTypeId();
			this.parent = parent;
			this.properties = properties;
		}

		/**
		 * @return typeId of the component to which this entry is attached via templates
		 */
		public String getTypeId() {
			return typeId;
		}

		/**
		 * @return type of the component to which this entry is attached via templates
		 */
		public String getType() {
			return type;
		}

		/**
		 * @return list of sub entries of this entry if this is component entry, empty list otherwise
		 */
		public List<TreeEntry> getSubEntries() {
			return CollectionsUtils.unmodifiableList(subEntries);
		}

		/**
		 * Adds given sub entries
		 * @param newEntries to be added
		 */
		public void addSubEntries(List<TreeEntry> newEntries) {
			if (!(this.subEntries instanceof ArrayList)) {
				this.subEntries = new ArrayList<>(newEntries.size());
			}
			this.subEntries.addAll(newEntries);
		}

		/**
		 * Removes given sub entries
		 * @param entries to be removed
		 */
		public void removeSubEntries(List<TreeEntry> entries) {
			this.subEntries.removeAll(entries);
		}

		/**
		 * @return parent entry when this is template entry, {@code null} when this is component entry
		 */
		public @Nullable TreeEntry getParent() {
			return parent;
		}

		/**
		 * @return properties of the template or {@code null} when this is the component entry
		 */
		public @Nullable ComponentTemplateProperties getProperties() {
			return properties;
		}

		/**
		 * @return {@code true} when entry is checked by user, {@code false} otherwise
		 */
		public boolean isChecked() {
			return isChecked;
		}

		/**
		 * Sets checked state
		 * @param isChecked {@code true} when entry was checked by user, {@code false} otherwise
		 */
		public void setChecked(boolean isChecked) {
			this.isChecked = isChecked;
		}
	}

	/** Number of rows that will be initially shown in viewer */
	private static final int VIEWER_INITIAL_ROWS_COUNT = 20;

	/** List of all root entries in viewer */
	private ArrayList<TreeEntry> entries = new ArrayList<>();
	/** Currently selected entry in viewer*/
	@Nullable TreeEntry currentlySelectedEntry = null;
	/** Viewer with entries which represent templates */
	private @Nullable TreeViewer viewer;
	/** Button for removing selected use cases */
	private @Nullable Button removeSelectedUseCaseButton;
	/** Button for exporting selected use cases */
	private @Nullable Button exportSelectedUseCaseButton;
	/** Button for opening details dialog */
	private @Nullable Button detailsButton;

	/**
	 * Constructor
	 * @param parentShell shell in which this dialog will be opened
	 */
	protected ComponentTemplateManagerDialog(Shell parentShell) {
		super(parentShell);
	}

	/**
	 * Opens the dialog
	 * @param shell in which the dialog should be created
	 */
	public static void open(Shell shell) {
		ComponentTemplateManagerDialog dialog = new ComponentTemplateManagerDialog(shell);
		dialog.setBlockOnOpen(true);
		dialog.open();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
	 */
	@Override
	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText(Messages.get().ComponentTemplateManagerDialog_Title);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createButton(org.eclipse.swt.widgets.Composite, int, java.lang.String, boolean)
	 */
	@Override
	protected @Nullable Button createButton(@Nullable Composite parent, int id, @Nullable String label, boolean defaultButton) {
		if (id == IDialogConstants.CANCEL_ID) {
			return null;
		}
		return super.createButton(parent, id, label, defaultButton);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected Control createDialogArea(Composite parent) {
		Composite composite = (Composite) super.createDialogArea(parent);
		GridLayout layout = new GridLayout(1, false);
		composite.setLayout(layout);
		Composite contentComposite = new Composite(composite, SWT.NONE);
		GridLayout contentCompositeLayout = new GridLayout(2, false); // left and right parts
		contentComposite.setLayout(contentCompositeLayout);
		contentComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		createAndPrepareTreeViewer(contentComposite);
		createButtons(composite);
		updateButtons();
		Button detailsButtonLoc = detailsButton;
		if (detailsButtonLoc != null) {
			detailsButtonLoc.setEnabled(false);
		}
		return composite;
	}

	/**
	 * Updates which entry is currently selected for details dialog
	 * @param entry to be selected
	 */
	void updateDetailsSelection(TreeEntry entry) {
		currentlySelectedEntry = entry;
	}

	/**
	 * Creates and prepares viewer in given composite
	 * @param contentComposite in which the viewer's composite will be created
	 */
	private void createAndPrepareTreeViewer(Composite contentComposite) {
		Composite treeComposite = new Composite(contentComposite, SWT.NONE);
		GridLayout treeCompositeLayout = new GridLayout(1, false);
		treeComposite.setLayout(treeCompositeLayout);
		treeComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		PatternFilter patternFilter = new PatternFilter(); // FIXME TomasR v13 maintenance - Extract from AddComponentDialog to common class that filters all columns
		@SuppressWarnings("deprecation") // Required by RAP
		FilteredTree filteredTree = new FilteredTree(treeComposite, SWT.BORDER | SWT.SINGLE | SWT.FULL_SELECTION , patternFilter, true);
		TreeViewer treeViewer = filteredTree.getViewer();
		assert treeViewer != null : "TreeViewer was not created in FilteredTree"; //$NON-NLS-1$
		SWTFactoryProxy.INSTANCE.enableHtmlTooltipFor(treeViewer);
		viewer = treeViewer;
		Tree tree = treeViewer.getTree();
		SWTFactoryProxy.INSTANCE.setTestId(tree, TestIDs.PERIPHS_USECASES_MANAGER_TREE);
		tree.setHeaderVisible(true);
		tree.setLinesVisible(true);
		GridData treeData = new GridData(SWT.FILL, SWT.FILL, true, true);
		treeData.heightHint = tree.getItemHeight() * VIEWER_INITIAL_ROWS_COUNT;
		tree.setLayoutData(treeData);
		treeViewer.addDoubleClickListener(new IDoubleClickListener() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
			 */
			@Override
			public void doubleClick(@Nullable DoubleClickEvent event) {
				openDetailsDialog(treeViewer.getTree().getShell());
			}
		});
		treeViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
			 */
			@Override
			public void selectionChanged(@Nullable SelectionChangedEvent event) {
				updateEnabledStateOfDetailsButton();
			}
		});
		TreeViewerColumn checkboxColumn = new TreeViewerColumn(treeViewer, SWT.NONE);
		checkboxColumn.getColumn().setText(Messages.get().ComponentTemplateManagerDialog_ViewerColumnSelectHeader);
		checkboxColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public @Nullable String getText(Object element) {
				return null;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getImage(java.lang.Object)
			 */
			@Override
			public @Nullable Image getImage(Object element) {
				if (element instanceof TreeEntry) {
					TreeEntry treeEntry = (TreeEntry) element;
					if (treeEntry.getProperties() != null) {
						return ToolsImages.getImage(treeEntry.isChecked() ? IToolsImages.ICON_CHECKED : IToolsImages.ICON_UNCHECKED);
					}
				}
				return null;
			}
		});
		checkboxColumn.setEditingSupport(new EditingSupport(treeViewer) {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.EditingSupport#setValue(java.lang.Object, java.lang.Object)
			 */
			@Override
			protected void setValue(Object element, @Nullable Object value) {
				// Intentionally empty
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.EditingSupport#getValue(java.lang.Object)
			 */
			@Override
			protected @Nullable Object getValue(Object element) {
				return null;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.EditingSupport#getCellEditor(java.lang.Object)
			 */
			@Override
			protected @Nullable CellEditor getCellEditor(Object element) {
				if (element instanceof TreeEntry) {
					TreeEntry treeEntry = (TreeEntry) element;
					if (treeEntry.getProperties() != null) {
						treeEntry.setChecked(!treeEntry.isChecked());
						updateButtons();
					}
					treeViewer.update(element, null);
				}
				return null;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.EditingSupport#canEdit(java.lang.Object)
			 */
			@Override
			protected boolean canEdit(Object element) {
				return true;
			}
		});
		TreeViewerColumn nameColumn = new TreeViewerColumn(treeViewer, SWT.NONE);
		nameColumn.getColumn().setText(Messages.get().ComponentTemplateManagerDialog_ViewerColumnNameHeader);
		nameColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public @Nullable String getText(Object element) {
				if (element instanceof TreeEntry) {
					TreeEntry treeEntry = (TreeEntry) element;
					ComponentTemplateProperties properties = treeEntry.getProperties();
					if (properties == null) {
						return treeEntry.getType();
					} else {
						return properties.getName();
					}
				}
				return null;
			}
		});
		TreeViewerColumn descriptionColumn = new TreeViewerColumn(treeViewer, SWT.NONE);
		descriptionColumn.getColumn().setText(Messages.get().ComponentTemplateManagerDialog_ViewerColumnDescriptionHeader);
		descriptionColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public @Nullable String getText(Object element) {
				if (element instanceof TreeEntry) {
					TreeEntry treeEntry = (TreeEntry) element;
					ComponentTemplateProperties properties = treeEntry.getProperties();
					if (properties != null) {
						return properties.getDescription();
					}
				}
				return null;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.CellLabelProvider#getToolTipText(java.lang.Object)
			 */
			@Override
			public @Nullable String getToolTipText(Object element) {
				if (element instanceof TreeEntry) {
					TreeEntry treeEntry = (TreeEntry) element;
					ComponentTemplateProperties properties = treeEntry.getProperties();
					if (properties != null) {
						return properties.getDescription();
					}
				}
				return null;
			}
		});
		TreeViewerColumn sdkCompColumn = new TreeViewerColumn(treeViewer, SWT.NONE);
		sdkCompColumn.getColumn().setText(Messages.get().ComponentTemplateManagerDialog_ViewerColumnSDKComponentHeader);
		sdkCompColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public @Nullable String getText(Object element) {
				if (element instanceof TreeEntry) {
					TreeEntry treeEntry = (TreeEntry) element;
					ComponentTemplateProperties properties = treeEntry.getProperties();
					if (properties != null) {
						return AddComponentDialog.buildDriverDependencyInfo(true, properties.getSwCompRefs());
					}
				}
				return null;
			}
		});
		TreeViewerColumn typeIdColumn = new TreeViewerColumn(treeViewer, SWT.NONE);
		typeIdColumn.getColumn().setText(Messages.get().ComponentTemplateManagerDialog_ViewerColumnTypeIdHeader);
		typeIdColumn.setLabelProvider(new ColumnLabelProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ColumnLabelProvider#getText(java.lang.Object)
			 */
			@Override
			public @Nullable String getText(Object element) {
				if (element instanceof TreeEntry) {
					TreeEntry treeEntry = (TreeEntry) element;
					return treeEntry.getTypeId();
				}
				return null;
			}
		});
		treeViewer.setContentProvider(new ITreeContentProvider() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
			 */
			@Override
			public boolean hasChildren(Object element) {
				if (element instanceof TreeEntry) {
					return !((TreeEntry) element).getSubEntries().isEmpty();
				}
				return false;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
			 */
			@Override
			public @Nullable Object getParent(Object element) {
				if (element instanceof TreeEntry) {
					return ((TreeEntry) element).getParent();
				}
				return null;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ITreeContentProvider#getElements(java.lang.Object)
			 */
			@Override
			public Object[] getElements(Object inputElement) {
				if (inputElement instanceof List) {
					List<?> inputList = (List<?>) inputElement;
					return inputList.toArray(new @NonNull Object @NonNull[inputList.size()]);
				}
				return new Object[0];
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
			 */
			@Override
			public Object[] getChildren(Object parentElement) {
				if (parentElement instanceof TreeEntry) {
					List<TreeEntry> subEntries = ((TreeEntry) parentElement).getSubEntries();
					return subEntries.toArray(new @NonNull TreeEntry[subEntries.size()]);
				}
				return new Object[0];
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
			 */
			@Override
			public void inputChanged(Viewer changedViewer, @Nullable Object oldInput, @Nullable Object newInput) {
				// Do nothing - required for RAP
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
			 */
			@Override
			public void dispose() {
				// Do nothing - required for RAP
			}
		});
		treeViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
			 */
			@Override
			public void selectionChanged(@Nullable SelectionChangedEvent event) {
				if (event == null) {
					return;
				}
				ISelection selection = event.getSelection();
				if (selection instanceof StructuredSelection) {
					StructuredSelection structureSelection = (StructuredSelection) selection;
					Object firstElement = structureSelection.getFirstElement();
					if (firstElement instanceof TreeEntry) {
						updateDetailsSelection((TreeEntry) firstElement);
					}
				}
			}
		});
		ArrayList<TreeEntry> entriesLoc = prepareEntries();
		entries = entriesLoc;
		treeViewer.setInput(entriesLoc);
		treeViewer.expandAll(); // FIXME TomasR v13 maintenance - This operation could be expensive in GUI when dozens of use cases are present. Consider constant widths instead.
		for (TreeColumn column : tree.getColumns()) {
			column.pack();
		}
		treeViewer.collapseAll();
	}

	/**
	 * Updates buttons
	 */
	protected void updateButtons() {
		Button exportSelectedUseCaseButtonLoc = exportSelectedUseCaseButton;
		Button removeSelectedUseCaseButtonLoc = removeSelectedUseCaseButton;
		List<ComponentTemplateProperties> selectedUseCases = collectSelectedUseCases();
		boolean enabled = !selectedUseCases.isEmpty();
		if (exportSelectedUseCaseButtonLoc != null) {
			exportSelectedUseCaseButtonLoc.setEnabled(enabled);
		}
		if (removeSelectedUseCaseButtonLoc != null) {
			removeSelectedUseCaseButtonLoc.setEnabled(enabled);
		}
	}

	/**
	 * Updates enabled state of details button based on selected item in viewer
	 */
	public void updateEnabledStateOfDetailsButton() {
		TreeViewer viewerLoc = viewer;
		Button detailsButtonLoc = detailsButton;
		if (viewerLoc == null) {
			return;
		}
		if (detailsButtonLoc == null) {
			return;
		}
		ISelection selection = viewerLoc.getSelection();
		if (selection instanceof StructuredSelection) {
			StructuredSelection sel = (StructuredSelection) selection;
			Object firstElement = sel.getFirstElement();
			if (firstElement instanceof TreeEntry) {
				TreeEntry entry = (TreeEntry) firstElement;
				ComponentTemplateProperties properties = entry.getProperties();
				detailsButtonLoc.setEnabled(properties != null);
			}
		}
	}

	/**
	 * Prepares entries of viewer.
	 * Returned list contains component entries which already contains template entries
	 * @return list of component entries
	 */
	private static ArrayList<TreeEntry> prepareEntries() {
		ArrayList<TreeEntry> entries = new ArrayList<>();
		Map<String, Map<String, ComponentTemplateProperties>> allTemplateProperties = AvailableComponents.loadAllTemplateProperties(ComponentTemplate.getComponentTemplatesFolder());
		for (Entry<String, Map<String, ComponentTemplateProperties>> entry : allTemplateProperties.entrySet()) {
			String typeId = entry.getKey();
			Collection<ComponentTemplateProperties> propertiesCollection = entry.getValue().values();
			if (!propertiesCollection.isEmpty()) {
				@SuppressWarnings("null") // Already checked for emptiness
				ComponentTemplateProperties firstTemplate = propertiesCollection.iterator().next();
				TreeEntry componentEntry = new TreeEntry(firstTemplate.getType(), typeId);
				for (ComponentTemplateProperties properties : propertiesCollection) {
					componentEntry.addSubEntries(Arrays.asList(new TreeEntry(componentEntry, properties)));
				}
				entries.add(componentEntry);
			}
		}
		return entries;
	}

	/**
	 * Collects properties of component templates which have been checked by user
	 * @return list of properties that has been checked in viewer
	 */
	List<ComponentTemplateProperties> collectSelectedUseCases() {
		ArrayList<ComponentTemplateProperties> selectedUseCases = new ArrayList<>();
		for (TreeEntry componentEntry : entries) {
			List<TreeEntry> useCaseEntries = componentEntry.getSubEntries();
			for (TreeEntry useCaseEntry : useCaseEntries) {
				ComponentTemplateProperties properties = useCaseEntry.getProperties();
				if (properties != null && useCaseEntry.isChecked()) {
					selectedUseCases.add(properties);
				}
			}
		}
		return selectedUseCases;
	}

	/**
	 * Imports templates with given names from given folder
	 * @param folder from where the use cases will be imported
	 * @param fileNames of the use case files to import
	 * @return list of use case properties
	 */
	private static List<ComponentTemplateProperties> importTemplatesInternal(String folder, List<String> fileNames) {
		AvailableComponents availableComponents = Controller.getInstance().getMcu().getAvailableComponents();
		ArrayList<ComponentTemplateProperties> newProperties = new ArrayList<>(fileNames.size());
		for (String fileName : fileNames) {
			File file = new File(folder, fileName);
			if (!file.exists() || file.isDirectory()) {
				continue;
			}
			ComponentTemplate template = ComponentTemplate.loadTemplateFromFile(file);
			if (template != null) {
				ComponentTemplateProperties importTemplate = ComponentTemplate.importTemplate(template, availableComponents);
				if (importTemplate != null) {
					newProperties.add(importTemplate);
				}
			}
		}
		return newProperties;
	}

	/**
	 * Imports use cases of given names from given folder and updates viewer
	 * @param fileNames of the use case files to import
	 * @param folder from where the use cases will be imported
	 */
	public void importUseCases(List<String> fileNames, String folder) {
		List<ComponentTemplateProperties> newProperties = importTemplatesInternal(folder, fileNames);
		if (!newProperties.isEmpty()) {
			updateViewerContent(Collections.emptyList(), newProperties);
		}
	}

	/**
	 * Removes use cases that are selected in viewer and the viewer is then updated
	 */
	public void removeSelectedUseCases() {
		boolean approved = MessageDialog.openQuestion(getShell(), Messages.get().ComponentTemplateManagerDialog_ButtonRemoveChecked_ConfirmationTitle,
				Messages.get().ComponentTemplateManagerDialog_ButtonRemoveChecked_ConfirmationMessage);
		if (approved) {
			removedSelectedUseCasesInternal();
		}
	}

	/**
	 * Removes use cases that are selected in viewer and the viewer is then updated
	 */
	private void removedSelectedUseCasesInternal() {
		List<ComponentTemplateProperties> selectedUseCases = collectSelectedUseCases();
		AvailableComponents availableComponents = Controller.getInstance().getMcu().getAvailableComponents();
		for (ComponentTemplateProperties useCase : selectedUseCases) {
			ComponentTemplate.delete(useCase, availableComponents);
		}
		updateViewerContent(selectedUseCases, Collections.emptyList());
	}

	/**
	 * Exports given use cases to given location on disk
	 * @param useCases to be exported
	 * @param locationOnDisk where the use cases will be exported
	 */
	private static void exportUseCasesInternal(List<ComponentTemplateProperties> useCases, String locationOnDisk) {
		File exportDirectory = new File(locationOnDisk);
		if (!exportDirectory.exists()) {
			return;
		}
		if (!exportDirectory.isDirectory()) {
			return;
		}
		for (ComponentTemplateProperties properties : useCases) {
			ComponentTemplate.export(properties, exportDirectory);
		}
	}

	/**
	 * Exports selected use cases to folder with given path
	 * @param pathToFolder where use cases will be exported
	 */
	public void exportSelectedUseCases(String pathToFolder) {
		List<ComponentTemplateProperties> selectedUseCases = collectSelectedUseCases();
		if (selectedUseCases.isEmpty()) {
			return;
		}
		exportUseCasesInternal(selectedUseCases, pathToFolder);
	}

	/**
	 * Opens detail dialog for currently selected use case
	 * @param shell in which the details dialog will open
	 */
	public void openDetailsDialog(Shell shell) {
		TreeEntry currentlySelectedEntryLoc = currentlySelectedEntry;
		if (currentlySelectedEntryLoc == null) {
			return;
		}
		ComponentTemplateProperties properties = currentlySelectedEntryLoc.getProperties();
		if (properties == null) {
			return;
		}
		ComponentTemplateProperties dialogResult = ComponentTemplateDetailDialog.open(shell, properties);
		if (dialogResult != null) {
			updateViewerContent(Arrays.asList(properties), Arrays.asList(dialogResult));
		}
	}

	/**
	 * Creates composite with buttons in given composite
	 * @param composite in which to create composite with buttons
	 */
	private void createButtons(Composite composite) {
		Composite mainButtonsComposite = new Composite(composite, SWT.NONE);
		GridLayout mainButtonsCompositeLayout = new GridLayout(4, false); // 3 buttons
		mainButtonsComposite.setLayout(mainButtonsCompositeLayout);
		mainButtonsComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
		Button importUseCaseButton = new Button(mainButtonsComposite, SWT.PUSH);
		SWTFactoryProxy.INSTANCE.setTestId(importUseCaseButton, TestIDs.PERIPHS_ACTION_IMPORT_USECASES);
		importUseCaseButton.setText(Messages.get().ComponentTemplateManagerDialog_ButtonImport);
		importUseCaseButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				FileDialog dialog = new FileDialog(getShell(), SWT.MULTI);
				dialog.setText(Messages.get().ComponentTemplateManagerDialog_ButtonImportDialogTitle);
				dialog.setFilterExtensions(new String[]{UtilsText.ASTERISK + UtilsText.DOT + ComponentTemplate.EXTENSION_OF_COMPONENT_TEMPLATE});
				String open = dialog.open();
				if (open == null) {
					return;
				}
				List<String> fileNames = Arrays.asList(dialog.getFileNames());
				File firstFile = new File(open);
				String folder = firstFile.getParent();
				if (folder == null) {
					return;
				}
				importUseCases(fileNames, folder);
			}
		});
		Button removeSelectedUseCaseButtonLoc = new Button(mainButtonsComposite, SWT.PUSH);
		removeSelectedUseCaseButton = removeSelectedUseCaseButtonLoc;
		SWTFactoryProxy.INSTANCE.setTestId(removeSelectedUseCaseButtonLoc, TestIDs.PERIPHS_ACTION_REMOVE_USECASES);
		removeSelectedUseCaseButtonLoc.setText(Messages.get().ComponentTemplateManagerDialog_ButtonRemoveChecked);
		removeSelectedUseCaseButtonLoc.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				removeSelectedUseCases();
			}
		});
		Button exportSelectedUseCaseButtonLoc = new Button(mainButtonsComposite, SWT.PUSH);
		exportSelectedUseCaseButton = exportSelectedUseCaseButtonLoc;
		SWTFactoryProxy.INSTANCE.setTestId(exportSelectedUseCaseButtonLoc, TestIDs.PERIPHS_ACTION_EXPORT_USECASES);
		exportSelectedUseCaseButtonLoc.setText(Messages.get().ComponentTemplateManagerDialog_ButtonExportChecked);
		exportSelectedUseCaseButtonLoc.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Shell shell = getShell();
				if (shell == null) {
					return;
				}
				String pathForExport = SWTFactoryProxy.INSTANCE.openDirectoryDialog(shell, null, null, null);
				if (pathForExport == null) {
					return;
				}
				exportSelectedUseCases(pathForExport);
			}
		});
		Button detailsButtonLoc = new Button(mainButtonsComposite, SWT.PUSH);
		detailsButton = detailsButtonLoc;
		SWTFactoryProxy.INSTANCE.setTestId(detailsButtonLoc, TestIDs.PERIPHS_ACTION_DETAILS_OF_USECASE);
		detailsButtonLoc.setText(Messages.get().ComponentTemplateManagerDialog_ButtonDetailsOfSelected);
		detailsButtonLoc.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Shell shell = detailsButtonLoc.getShell();
				openDetailsDialog(shell);
			}
		});
	}

	/**
	 * Updates content of the viewer
	 * @param propertiesToRemove list of properties to be removed from viewer
	 * @param propertiesToAdd list of properties to be added to viewer
	 */
	public void updateViewerContent(List<ComponentTemplateProperties> propertiesToRemove, List<ComponentTemplateProperties> propertiesToAdd) {
		TreeViewer viewerLoc = viewer;
		if (viewerLoc == null) {
			return;
		}
		for (ComponentTemplateProperties properties : propertiesToRemove) {
			TreeEntry componentEntry = CollectionsUtils.nullableOptionalGet(entries.stream().filter(x -> x.getTypeId().equals(properties.getTypeId())).findAny());
			if (componentEntry != null) {
				List<TreeEntry> subEntries = new ArrayList<>(componentEntry.getSubEntries());
				List<TreeEntry> entriesToRemove = subEntries.stream().filter(entry -> entry.getProperties() != null && propertiesToRemove.contains(entry.getProperties())).collect(CollectorsUtils.toList());
				componentEntry.removeSubEntries(entriesToRemove);
			}
		}
		if (!propertiesToRemove.isEmpty()) {
			List<TreeEntry> emptyComponentEntris = entries.stream().filter(entry -> entry.getSubEntries().isEmpty()).collect(CollectorsUtils.toList());
			entries.removeAll(emptyComponentEntris);
		}
		for (ComponentTemplateProperties template : propertiesToAdd) {
			TreeEntry componentEntry = CollectionsUtils.nullableOptionalGet(entries.stream().filter(x -> x.getTypeId().equals(template.getTypeId())).findAny());
			if (componentEntry == null) {
				componentEntry = new TreeEntry(template.getType(), template.getTypeId());
				entries.add(componentEntry);
			}
			TreeEntry newEntry = new TreeEntry(componentEntry, template);
			componentEntry.addSubEntries(Arrays.asList(newEntry));
		}
		viewerLoc.refresh();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#isResizable()
	 */
	@Override
	protected boolean isResizable() {
		return true;
	}
}
