/**
 * Copyright 2021-2022 NXP
 * Created: 14 Aug 2021
 */
package com.nxp.swtools.periphs.gui.view;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.nxp.swtools.common.ui.utils.swt.FontFactory;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.widgets.InstantSearchList;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.NonNullByDefault;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.controller.MigrationHelper;
import com.nxp.swtools.periphs.controller.events.EventTypes;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.provider.configuration.SharedConfigurationFactory;
import com.nxp.swtools.provider.toolchainproject.IToolchainProjectWithSdk;
import com.nxp.swtools.resourcetables.model.config.IChildProvidable;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.resourcetables.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.resourcetables.model.data.IConfigurationComponent;
import com.nxp.swtools.resourcetables.model.data.SWComponent;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.preferences.KEPreferences;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsImages;

/**
 * Dialog for offering migration of component to version that is present in toolchain project
 * @author Tomas Rudolf - nxf31690
 */
@NonNullByDefault // This class is non null by default
public class MigrationOfferDialog extends Dialog {
	/** Offset to use when getting offered migrations based on combobox item index */
	private static final int OFFERED_MIGRATION_LABELS_OFFSET = 1;
	/** Label of of item that rejects migration */
	public static final String DO_NOT_MIGRATE_ITEM_LABEL = "Do not migrate"; //$NON-NLS-1$
	/** Logger of the class */
	private static final Logger LOGGER = LogManager.getLogger(MigrationOfferDialog.class);
	/** Associated view site */
	final Shell shell;
	/** The controller wrapper */
	final IControllerWrapper controllerWrapper;
	/** Button that invokes migration */
	private @Nullable Button migrateButton;
	/** Result of this dialog */
	private List<IChildProvidable> result = Collections.emptyList();
	/** Map of combobox for each current component that contains components to which user can migrate */
	Map<ConfigurationComponentTypeId, InstantSearchList> replacementComboboxes = new HashMap<>();
	/** Map of checkbox for each current component that represent permanent rejection of migrations for that component */
	Map<ConfigurationComponentTypeId, Button> rejectionCheckBoxes = new HashMap<>();
	/** List of component pairs <current, replacement> that are offered by this dialog */
	private Map<ConfigurationComponentTypeId, List<ConfigurationComponentTypeId>> offeredComponents = new HashMap<>();
	/** Flag whether all component migration offers should be displayed **/
	private boolean showAllComponents = false;

	/**
	 * Constructor.
	 * @param shell in which the dialog will open
	 * @param instance from which to migrate
	 * @param controllerWrapper containing the generic controller
	 */
	private MigrationOfferDialog(Shell shell, IControllerWrapper controllerWrapper) {
		super(shell);
		this.shell = shell;
		this.controllerWrapper = controllerWrapper;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected Control createDialogArea(Composite parent) {
		Composite dialogComposite = (Composite) super.createDialogArea(parent);
		dialogComposite.setLayout(new GridLayout());
		offeredComponents.clear();
		offeredComponents.putAll(MigrationHelper.getListOfComponentUpgrades(controllerWrapper.getController(), showAllComponents));
		createComponentsList(dialogComposite, offeredComponents);
		return dialogComposite;
	}

	/**
	 * Creates list of composites that contains the offered migration of component
	 * @param parent composite
	 * @param components Collection of component typeId pairs <current, replacement>
	 */
	private void createComponentsList(Composite parent, Map<ConfigurationComponentTypeId, List<ConfigurationComponentTypeId>> components) {
		// Components composite
		Composite rowComposite = new Composite(parent, SWT.NONE);
		rowComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		rowComposite.setLayout(new GridLayout(5, false));
		// Header label
		if (!isToolchainProjectDetected()) {
			Label warningImage = new Label(rowComposite, SWT.NONE);
			warningImage.setImage(ToolsImages.getStatusIcon(ErrorLevels.LEVEL_WARNING));
			warningImage.setLayoutData(new GridData(SWT.LEFT, SWT.TOP, false, true, 1, 1));
			Label warningLabel = new Label(rowComposite, SWT.WRAP);
			GridData layoutData = new GridData(SWT.CENTER, SWT.TOP, false, true, 4, 1);
			layoutData.widthHint = 300;
			warningLabel.setLayoutData(layoutData);
			warningLabel.setText(Messages.get().OpenMigrationOfferDialogHandler_Warning);
		}
		Label permanentIgnoreLabel = new Label(rowComposite, SWT.NONE);
		permanentIgnoreLabel.setLayoutData(new GridData(SWT.RIGHT, SWT.TOP, true, true, 5, 1));
		permanentIgnoreLabel.setText(Messages.get().MigrationOfferDialog_IgnoreColumnLabel);
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(permanentIgnoreLabel, Messages.get().MigrationOfferDialog_IgnoreColumnTooltip);
		Set<String> rejectedComponents = MigrationHelper.getIgnoredComponents(this.controllerWrapper.getController());
		// Entry rows
		for (Entry<ConfigurationComponentTypeId, List<ConfigurationComponentTypeId>> pair : components.entrySet()) {
			ConfigurationComponentTypeId currentComponentTypeId = pair.getKey();
			IConfigurationComponent currentComponent = currentComponentTypeId.getConfigurationComponent();
			boolean isRejected = rejectedComponents.contains(currentComponent.getId());
			List<ConfigurationComponentTypeId> newComponents = pair.getValue();
			// Label of current component
			Label currentComponentNameLabel = new Label(rowComposite, SWT.BOLD);
			currentComponentNameLabel.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, false, false, 1, 1));
			String currentComponentName = currentComponent.getUINameString();
			currentComponentNameLabel.setText(currentComponentName);
			FontFactory.changeStyle(currentComponentNameLabel, SWT.BOLD);
			// Version of current component
			Label currentVersionLabel = new Label(rowComposite, SWT.NONE);
			currentVersionLabel.setLayoutData(new GridData(SWT.CENTER, SWT.CENTER, false, false));
			// FIXME TomasR v13 - PH to decide how to present component with multiple required SDK components. Draft is to then show SDK component ID with version on multiple rows
			currentVersionLabel.setText(currentComponent.getComponents().get(0).getVersionStr());
			// Arrow
			Label arrowLabel = new Label(rowComposite, SWT.NONE);
			arrowLabel.setLayoutData(new GridData(SWT.CENTER, SWT.CENTER, false, false));
			arrowLabel.setImage(ToolsImages.getImage(IToolsImages.ICON_RIGHT));
			// Combobox with selection of migration
			InstantSearchList combo = new InstantSearchList(rowComposite, SWT.BORDER);
			replacementComboboxes.put(currentComponentTypeId, combo);
			List<String> items = new ArrayList<>(1);
			items.add(DO_NOT_MIGRATE_ITEM_LABEL);
			for (ConfigurationComponentTypeId component : newComponents) {
				StringBuilder builder = new StringBuilder();
				builder.append(component.getConfigurationComponent().getUINameString()); // FIXME TomasR v13 maintenance - Use expression to resolve label of the component?
				builder.append(UtilsText.LEFT_BRACKET);
				for (SWComponent swComp : component.getComponentReferences()) {
					builder.append(swComp.getVersionStr());
					List<String> revisionsList = swComp.getRevisionsList();
					if (!revisionsList.isEmpty()) {
						builder.append(UtilsText.COMMA_SPACE);
						builder.append(MessageFormat.format(Messages.get().AddComponentDialog_RevisionsList_Prefix, UtilsText.join(revisionsList, UtilsText.COMMA)));
					}
				}
				builder.append(UtilsText.RIGHT_BRACKET);
				items.add(builder.toString());
			}
			combo.setItems(items.toArray(new @NonNull String[items.size()]));
			int defaultItemIndex = isToolchainProjectDetected() ? 1 : 0;
			combo.setText(items.get(defaultItemIndex));
			combo.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
			SWTFactoryProxy.INSTANCE.setTestId(combo, TestIDs.PERIPHS_MIGRATION_COMBO + currentComponentName);
			if (isRejected) {
				combo.setEnabled(false);
				combo.setText(DO_NOT_MIGRATE_ITEM_LABEL);
			}
			// Checkbox whether the component should be permanently ignored
			Button permanentIgnoreCheckbox = new Button(rowComposite, SWT.CHECK);
			permanentIgnoreCheckbox.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false));
			rejectionCheckBoxes.put(currentComponentTypeId, permanentIgnoreCheckbox);
			SWTFactoryProxy.INSTANCE.setTestId(permanentIgnoreCheckbox, TestIDs.PERIPHS_MIGRATION_PERMANENT_IGNORE_CHECKBOX + currentComponentName);
			permanentIgnoreCheckbox.addSelectionListener(new SelectionAdapter() {
				/* (non-Javadoc)
				 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
				 */
				@Override
				public void widgetSelected(@NonNull SelectionEvent e) {
					boolean ignore = permanentIgnoreCheckbox.getSelection();
					if (ignore) {
						combo.setEnabled(false);
						combo.setText(DO_NOT_MIGRATE_ITEM_LABEL);
					} else {
						combo.setEnabled(true);
					}
				}
			});
			permanentIgnoreCheckbox.setSelection(isRejected); // Rejected components will have checked checkbox
		}
	}

	/**
	 * @return {@code true} when toolchain project was detected, {@code false} otherwise
	 */
	private static boolean isToolchainProjectDetected() {
		IToolchainProjectWithSdk toolchainProject = SharedConfigurationFactory.getSharedConfigurationSingleton().getToolchainProject();
		return ((toolchainProject != null) && toolchainProject.wasProjectDetected());
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	@Override
	protected void okPressed() {
		boolean savedAnyRejection = saveRejectionsOfMigrationOffer(!showAllComponents);
		Optional<Boolean> anyMigrationPerformed = controllerWrapper.getController().runTransaction(() -> {
			boolean migrationPerformed = false;
			for (Entry<ConfigurationComponentTypeId, List<ConfigurationComponentTypeId>> entry : offeredComponents.entrySet()) {
				ConfigurationComponentTypeId currentComponentTypeId = entry.getKey();
				InstantSearchList combobox = replacementComboboxes.get(currentComponentTypeId);
				if (combobox != null) {
					String comboBoxText = combobox.getText();
					if (DO_NOT_MIGRATE_ITEM_LABEL.equals(comboBoxText)) {
						continue; // Migration is not wanted
					}
					int selectedReplacementComponentIndex = combobox.getItems().indexOf(comboBoxText);
					if (selectedReplacementComponentIndex == -1) {
						LOGGER.log(Level.SEVERE, "[TOOL][{0}] Replacement component combobox does not contain the selected value. The component cannot be chosen"); //$NON-NLS-1$
						continue;
					}
					int indexOfNewComponentTypeId = selectedReplacementComponentIndex - OFFERED_MIGRATION_LABELS_OFFSET;
					ConfigurationComponentTypeId newTypeId = entry.getValue().get(indexOfNewComponentTypeId);
					List<IComponentInstanceConfig> instances = controllerWrapper.getController().getProfile().getAllInstances(currentComponentTypeId.getType());
					for (IComponentInstanceConfig instance : instances) {
						MigrationHelper.migrate(instance, newTypeId, controllerWrapper.getController(), MigrationOfferDialog.class);
						migrationPerformed = true;
					}
				}
			}
			return Boolean.valueOf(migrationPerformed);
		});
		if (anyMigrationPerformed.isPresent() && !anyMigrationPerformed.get().booleanValue() && savedAnyRejection) {
			controllerWrapper.getController().handleSettingChange(EventTypes.CHANGE, MigrationOfferDialog.class, Messages.get().MigrationOfferDialog_SavingRejections_Action,
					true, null, false);
		}
		super.okPressed();
	}

	/**
	 * Saves the choice to permanently reject the migration offers for components that have that checkbox checked
	 * @param includePrevious {@code true} when previous set of ignored components should be included, {@code false} otherwise
	 * @return {@code true} when there is at least one component that was saved as permanently ignored, {@code false} otherwise
	 */
	private boolean saveRejectionsOfMigrationOffer(boolean includePrevious) {
		Set<String> previouslyIgnoredComponents = MigrationHelper.getIgnoredComponents(controllerWrapper.getController());
		Set<String> newIgnoredComponents = new HashSet<>();
		for (Entry<ConfigurationComponentTypeId, Button> entry : rejectionCheckBoxes.entrySet()) {
			if (entry.getValue().getSelection()) {
				newIgnoredComponents.add(entry.getKey().getType());
			}
		}
		if (includePrevious) {
			newIgnoredComponents.addAll(previouslyIgnoredComponents);
		}
		MigrationHelper.addIgnoredComponents(controllerWrapper.getController(), newIgnoredComponents);
		return !CollectionsUtils.difference(previouslyIgnoredComponents, newIgnoredComponents, new ArrayList<>(), new ArrayList<>());
	}

	/**
	 * Checks whether there are migrations to be offered and if migration offers are allowed
	 * @param wrapper with the controller to be used
	 * @return {@code true} when migrations are allowed and there is at lease one component that can be migrated to different version (excluding rejected), {@code false} otherwise
	 */
	private static boolean shouldMigrationsBeOffered(IControllerWrapper wrapper) {
		if (!KEPreferences.areComponentMigrationsAllowed()) {
			return false;
		}
		return !MigrationHelper.getListOfComponentUpgrades(wrapper.getController(), false).isEmpty();
	}

	/**
	 * Checks whether there are migrations to be offered and if migration offers are allowed. Includes rejected migration offers
	 * @param wrapper with the controller to be used
	 * @return {@code true} when migrations are allowed and there is at lease one component that can be migrated to different version (including rejected), {@code false} otherwise
	 */
	public static boolean canDialogBeOpened(IControllerWrapper wrapper) {
		if (!KEPreferences.areComponentMigrationsAllowed()) {
			return false;
		}
		return !MigrationHelper.getListOfComponentUpgrades(wrapper.getController(), true).isEmpty();
	}

	/**
	 * Checks whether the dialog should open automatically or not
	 * @param wrapper with the controller to be used
	 * @return {@code true} when toolchain project/IDE was detected, migration offers are allowed and there are components that can be migrated
	 */
	public static boolean shouldMigrationOfferDialogopenAutomatically(IControllerWrapper wrapper) {
		if (!isToolchainProjectDetected()) {
			return false;
		}
		return shouldMigrationsBeOffered(wrapper);
	}

	/**
	 * Open MigrateComponent dialog without component migration offers which were previously rejected
	 * @param shell in which to open the dialog
	 * @param instance to migrate from
	 * @param controllerWrapper wrapper with controllers to work with
	 * @return dialog instance
	 */
	public static MigrationOfferDialog open(Shell shell, IControllerWrapper controllerWrapper) {
		return open(shell, controllerWrapper, false);
	}

	/**
	 * Open MigrateComponent dialog
	 * @param shell in which to open the dialog
	 * @param instance to migrate from
	 * @param controllerWrapper wrapper with controllers to work with
	 * @param showAllComponents {@code true} when all component migration offers should be displayed,
	 * {@code false} when previously rejected component migration offers should not be displayed
	 * @return dialog instance
	 */
	public static MigrationOfferDialog open(Shell shell, IControllerWrapper controllerWrapper, boolean showAllComponents) {
		MigrationOfferDialog dialog = new MigrationOfferDialog(shell, controllerWrapper);
		dialog.setBlockOnOpen(true);
		dialog.setShowAllComponents(showAllComponents);
		dialog.open();
		return dialog;
	}

	/**
	 * Sets flag whether to show the component migration offers which were previously rejected
	 * @param showAllComponents {@code true} when all component migration offers should be displayed,
	 * {@code false} when the previously rejected component migration offers should be hidden
	 */
	private void setShowAllComponents(boolean showAllComponents) {
		this.showAllComponents = showAllComponents;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#cancelPressed()
	 */
	@Override
	protected void cancelPressed() {
		close();
	}

	/**
	 * Get result of dialog.
	 * @return list of instances that were migrated
	 */
	public List<IChildProvidable> getResult() {
		return result;
	}

	/**
	 * Set result of dialog.
	 * @param result list of instances that were migrated
	 */
	void setResult(List<IChildProvidable> result) {
		this.result = result;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
	 */
	@Override
	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText(Messages.get().MigrationOfferDialog_DialogTitle);
		setShellStyle(SWT.TITLE);
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#isResizable()
	 */
	@Override
	protected boolean isResizable() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.Dialog#createButtonsForButtonBar(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		migrateButton = createButton(parent, IDialogConstants.OK_ID, Messages.get().ComponentsView_MigrateComponentDialog_OK, true);
		SWTFactoryProxy.INSTANCE.setTestId(migrateButton, TestIDs.PERIPHS_MIGRATE_COMPONENT_SHELL_OK_BUTTON);
		createButton(parent, IDialogConstants.CANCEL_ID, com.nxp.swtools.common.ui.utils.Messages.get().MessageBoxDialog_Cancel, false);
	}
}
