/**
 * Copyright 2021 NXP
 * Created: Jan 19, 2021
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.core.runtime.IPath;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.WorkbenchException;
import org.eclipse.ui.XMLMemento;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.kex.MfactConstants;
import com.nxp.swtools.provider.configuration.ConfigChangeReason;
import com.nxp.swtools.provider.configuration.ISharedConfiguration;
import com.nxp.swtools.provider.configuration.SharedConfigurationAdapter;
import com.nxp.swtools.provider.configuration.SharedConfigurationFactory;
import com.nxp.swtools.resourcetables.model.config.ChildContext;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.resourcetables.model.config.IFunctionalGroup;
import com.nxp.swtools.utils.storage.StorageHelper;

/**
 * Helper class for remembering which collapsible setting was collapsed before
 * @author Tomas Rudolf - nxf31690
 */
public class CollapsibleSettingsStorageHelper {
	/** Logger of class */
	private static final @NonNull Logger LOGGER = LogManager.getLogger(CollapsibleSettingsStorageHelper.class);
	/** Name of the states element */
	private static final String STATES_ELEMENT_NAME = "states"; //$NON-NLS-1$
	/** Name of the state element */
	private static final String STATE_ELEMENT_NAME = "state"; //$NON-NLS-1$
	/** Name of the ID attribute */
	private static final String ID_ATTRIBUTE_NAME = "id"; //$NON-NLS-1$
	/** Name of the value attribute */
	private static final String VALUE_ATTRIBUTE_NAME = "value"; //$NON-NLS-1$
	/** Value which represents collapsed state */
	private static final boolean COLLAPSED_VALUE = false;
	/** Value which represent expanded state */
	private static final boolean EXPANDED_VALUE = true;
	/** editors content */
	public static final @NonNull String COLLAPSIBLE_SETTINGS_KEY = "collapsible_settings_"; //$NON-NLS-1$
	/** Storage for XML memento */
	@NonNull StorageHelper storageHelper = new StorageHelper(MfactConstants.TOOL_PERIPHERALS_ID);
	/** Map of expanded states for ids */
	private @NonNull Map<@NonNull String, @NonNull Boolean> collapsibleSettingsState = new HashMap<>();
	/** Location of current MEX */
	private @NonNull String mexFileLocation = UtilsText.EMPTY_STRING;

	/**
	 * Get singleton
	 * @return singleton of this class
	 */
	public static @NonNull CollapsibleSettingsStorageHelper getInstance() {
		try {
			CollapsibleSettingsStorageHelper helper = SWTFactoryProxy.INSTANCE.getSingletonInstance(CollapsibleSettingsStorageHelper.class);
			return helper;
		} catch (InstantiationException | IllegalAccessException e) {
			throw new IllegalStateException("Cannot obtain instance of a component editor helper", e); //$NON-NLS-1$
		}
	}

	/**
	 * Constructor
	 */
	public CollapsibleSettingsStorageHelper() {
		// Reset temporary location before saving to mex
		storageHelper.saveString(COLLAPSIBLE_SETTINGS_KEY + UtilsText.EMPTY_STRING, UtilsText.EMPTY_STRING);
		SharedConfigurationFactory.getSharedConfigurationSingleton().addListener(new SharedConfigurationAdapter() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.provider.configuration.SharedConfigurationAdapter#configurationReloaded(com.nxp.swtools.provider.configuration.ISharedConfiguration, com.nxp.swtools.provider.configuration.ConfigChangeReason)
			 */
			@Override
			public void configurationReloaded(@NonNull ISharedConfiguration sharedConfig, @NonNull ConfigChangeReason reason) {
				switch (reason) {
					case LOAD_CONFIG: {
						mexFileLocation = getMexFilePath();
						loadStates();
						break;
					}
					case NEW_CONFIG: {
						saveStates();
						mexFileLocation = getMexFilePath();
						collapsibleSettingsState.clear();
						break;
					}
					default:
						// DO nothing
				}
			}
		});
	}

	/**
	 * Set given control expanded state to collapsed
	 * @param control which should be marked as collapsed
	 */
	public void collapseControl(@NonNull IChildControl control) {
		boolean result = setCollapsedState(control, COLLAPSED_VALUE);
		if (!result) {
			LOGGER.log(Level.FINE, "[TOOL] Given control {0} does not contain full path to the root and was not saved", control.getChild().getId()); //$NON-NLS-1$
		}
	}

	/**
	 * Set given control expanded state to expanded
	 * @param control which should be marked as expanded
	 */
	public void expandControl(@NonNull IChildControl control) {
		boolean result = setCollapsedState(control, EXPANDED_VALUE);
		if (!result) {
			LOGGER.log(Level.FINE, "[TOOL] Given control {0} does not contain full path to the root and was not saved", control.getChild().getId()); //$NON-NLS-1$
		}
	}

	/**
	 * Sets information about expanded state of the control to map of states
	 * @param control about which save the expanded state
	 * @param state the expanded state to be saved
	 * @return {@code true} when state is saved, {@code false} otherwise
	 */
	private boolean setCollapsedState(IChildControl control, boolean state) {
		String fullSettingId = getFullId(control);
		if (fullSettingId == null) {
			return false;
		}
		collapsibleSettingsState.put(fullSettingId, Boolean.valueOf(state));
		saveStates();
		return true;
	}

	/**
	 * Returns full id of given control or null in case of problems
	 * @param control to get full id from
	 * @return full id of the child in control, {@code null} when functional group or component instance cannot be found
	 */
	private static @Nullable String getFullId(IChildControl control) {
		IChild child = control.getChild();
		ChildContext childContext = child.getChildContext();
		StringBuilder builder = new StringBuilder(100);
		IFunctionalGroup functionalGroup = childContext.getFunctionalGroup();
		if (functionalGroup == null) {
			return null;
		}
		builder.append(functionalGroup.getName()).append(UtilsText.DOT);
		IComponentInstanceConfig instance = childContext.getComponentInstanceConfig();
		if (instance == null) {
			return null;
		}
		builder.append(instance.getName()).append(UtilsText.DOT);
		builder.append(child.getId());
		String fullSettingId = builder.toString();
		return fullSettingId;
	}

	/**
	 * Check if given control is expanded or not
	 * @param control to be checked
	 * @return {@code true} if full id cannot be computed, information about control is not saved or saved information marks control as expended, {@code false} otherwise
	 */
	public boolean isExpanded(@NonNull IChildControl control) {
		String fullId = getFullId(control);
		if (fullId == null) {
			return true;
		}
		Boolean expanded = collapsibleSettingsState.get(fullId);
		if (expanded == null) {
			return true;
		}
		return expanded.booleanValue();
	}

	/**
	 * Check if given control has saved state
	 * @param control to be checked
	 * @return {@code true} if expansion state of the control is saved, {@code false} otherwise
	 */
	public boolean isStored(@NonNull IChildControl control) {
		String fullId = getFullId(control);
		if (fullId == null) {
			return false;
		}
		Boolean expanded = collapsibleSettingsState.get(fullId);
		return (expanded != null);
	}

	/**
	 * Save current states to tool preferences
	 */
	private void saveStates() {
		StringWriter writer = new StringWriter();
		try {
			XMLMemento xmlMemento = XMLMemento.createWriteRoot(STATES_ELEMENT_NAME); 
			for (Entry<@NonNull String, @NonNull Boolean> entry : collapsibleSettingsState.entrySet()) {
				IMemento child = xmlMemento.createChild(STATE_ELEMENT_NAME);
				child.putString(ID_ATTRIBUTE_NAME, entry.getKey());
				child.putBoolean(VALUE_ATTRIBUTE_NAME, entry.getValue().booleanValue());
			}
			xmlMemento.save(writer);
			storageHelper.saveString(COLLAPSIBLE_SETTINGS_KEY + mexFileLocation, UtilsText.safeString(writer.toString()));
		} catch (IOException e) {
			LOGGER.log(Level.INFO, "[TOOL] File exception occured. Detail: {0}", e.getMessage()); //$NON-NLS-1$
		}
	}

	/**
	 * Load states of current MEX file from tool preferences
	 */
	private void loadStates() {
		try {
			if (UtilsText.isEmpty(mexFileLocation)) {
				return;
			}
			String mementoStrContent = storageHelper.loadString(COLLAPSIBLE_SETTINGS_KEY + mexFileLocation, UtilsText.EMPTY_STRING);
			if (UtilsText.EMPTY_STRING.equals(mementoStrContent)) {
				return;
			}
			StringReader reader = new StringReader(mementoStrContent);
			// Prepare memento from storage
			XMLMemento xmlMemento = XMLMemento.createReadRoot(reader);
			// Read all children
			IMemento[] children = xmlMemento.getChildren();
			collapsibleSettingsState.clear();
			for (IMemento child : children) {
				String id = UtilsText.safeString(child.getString(ID_ATTRIBUTE_NAME));
				Boolean value = child.getBoolean(VALUE_ATTRIBUTE_NAME);
				value = value != null ? value : Boolean.FALSE;
				collapsibleSettingsState.put(id, value);
			}
		} catch (WorkbenchException e) {
			LOGGER.log(Level.SEVERE, "[TOOL] XMLMemento exception occured. Detail: {0}", e.getMessage()); //$NON-NLS-1$
		}
	}

	/**
	 * Looks up for MEX location
	 * @return path to MEX file
	 */
	private static String getMexFilePath() {
		@Nullable IPath locationPath = SharedConfigurationFactory.getSharedConfigurationSingleton().getLocationPath();
		String projectLocation;
		if (locationPath == null) {
			projectLocation = UtilsText.EMPTY_STRING;
		} else {
			projectLocation = locationPath.toOSString();
		}
		return projectLocation;
	}

	/**
	 * Performs actions that are needed during shutdown
	 */
	public void onShutdown() {
		saveStates();
	}

	/**
	 * Removes all entries from tool preferences file which are containing the given pattern
	 * @param pattern - based on which the entries will be removed
	 */
	public void removeSettingStateEntries(String pattern) {
		collapsibleSettingsState.keySet().removeIf(fullId -> fullId.contains(pattern));
		saveStates();
	}
}
