/**
 * Copyright 2017-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.math.BigInteger;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;

import com.nxp.swtools.common.ui.utils.GcUtils;
import com.nxp.swtools.common.ui.utils.swt.ControlDecorationUtils;
import com.nxp.swtools.common.ui.utils.swt.FontFactory;
import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.ScrolledCompositeHelper;
import com.nxp.swtools.common.ui.utils.swt.widgets.InstantSearchList;
import com.nxp.swtools.common.ui.utils.swt.widgets.ToggleButton;
import com.nxp.swtools.common.ui.utils.views.GenericSelectionProvider;
import com.nxp.swtools.common.ui.utils.views.PartListener2Adapter;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.expression.ExpressionException;
import com.nxp.swtools.common.utils.expression.IContext;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.configuration.SwToolsProduct;
import com.nxp.swtools.core.service.scriptapi.db.IRegParentPeripheralAPI;
import com.nxp.swtools.core.service.scriptapi.db.IRegistersDatabaseAPI;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.controller.APeriphController;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.controller.events.EventTypes;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.controller.PeriphControllerWrapper;
import com.nxp.swtools.periphs.gui.view.DocumentationView;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType;
import com.nxp.swtools.periphs.gui.wrappers.ComponentInstanceWrapper;
import com.nxp.swtools.periphs.model.data.Categories;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.provider.configuration.storage.periphs.AStoragePeriphsComponent;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IChildProvidable;
import com.nxp.swtools.resourcetables.model.config.IComponentConfig;
import com.nxp.swtools.resourcetables.model.config.IComponentInstanceConfig;
import com.nxp.swtools.resourcetables.model.config.IConfigSetConfig;
import com.nxp.swtools.resourcetables.model.config.IFunctionalGroup;
import com.nxp.swtools.resourcetables.model.config.RegistersModelSingleton;
import com.nxp.swtools.resourcetables.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.resourcetables.model.data.MasterPeripheral;
import com.nxp.swtools.resourcetables.model.data.SettingOptions;
import com.nxp.swtools.resourcetables.model.validation.ValidationHelper;
import com.nxp.swtools.utils.Limits;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.events.IEventListener;
import com.nxp.swtools.utils.events.ToolEvent;
import com.nxp.swtools.utils.preferences.KEPreferences;
import com.nxp.swtools.utils.profiler.Profiler;
import com.nxp.swtools.utils.progress.ProgressUtils;
import com.nxp.swtools.utils.registers.RegistersView;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsColors.SwToolsColors;
import com.nxp.swtools.utils.resources.ToolsImages;
import com.nxp.swtools.utils.support.markdown.MarkDownSupport;
import com.nxp.swtools.utils.text.TextBoxHelper;
import com.nxp.swtools.utils.text.TextBoxHelper.Status;
import com.nxp.swtools.utils.tooltip.ToolTipableFormatter;
import com.nxp.swtools.utils.tooltip.ToolTipableImplementation;
import com.nxp.swtools.utils.tooltip.ToolTipableMarkdownDescriptionDecorator;
import com.nxp.swtools.utils.view.ToolView;
import com.nxp.swtools.validation.engine.IValidationProblem;
import com.nxp.swtools.validation.engine.ValidationEngineFactory;
import com.nxp.swtools.validation.engine.ValidationProblemListenerAdapter;

/**
 * Editor used for editing component instance settings.
 * @author Juraj Ondruska
 * @author Tomas Rudolf - nxf31690
 */
public class ComponentSettingView extends EditorViewBase {
	/** Logger of class */
	static final @NonNull Logger LOGGER = LogManager.getLogger(ComponentSettingView.class);
	/** Separator used to divide component type and component name in the secondary ID of the view */
	public static final @NonNull String SECONDARY_ID_NAME_TYPE_SEPARATOR = "/"; //$NON-NLS-1$
	/** Separator used to divide views identification and occurrence counter in the secondary ID of the view */
	public static final @NonNull String SECONDARY_ID_IDENTIFICATION_OCCURENCE_SEPARATOR = "|"; //$NON-NLS-1$
	/** Group 'type' */
	public static final @NonNull String SECONDARY_ID_GROUP_TYPE = "type"; //$NON-NLS-1$
	/** Group 'name' */
	public static final @NonNull String SECONDARY_ID_GROUP_NAME = "name"; //$NON-NLS-1$
	/** Group 'uuid' */
	public static final @NonNull String SECONDARY_ID_GROUP_UUID = "uuid"; //$NON-NLS-1$
	/** Group 'occurrence' */
	public static final @NonNull String SECONDARY_ID_GROUP_OCCURRENCE = "occurrence"; //$NON-NLS-1$
	/** Pattern to match secondary id and separate each part to named group (type, name, uuid, occurrence) */
	public static final @NonNull Pattern SECONDARY_ID_GROUPS_PATTERN = Pattern.compile("(((?<type>.*?)/(?<name>.*?))|(?<uuid>.*?))\\|(?<occurrence>.*)"); //$NON-NLS-1$
	/** Number of columns in the component view title ((Title + category), documentation link, lock button, enable button) */
	public static final int TITLE_ROW_COLS = 4;
	/** Number of columns in the component settings (name, peripheral, etc.) */
	public static final int COMPONENT_COLS = 2;
	/** Number of columns in the config set settings */
	public static final int CONFIG_SET_COLS = 3;
	/** Number of columns in the set settings */
	public static final int SET_COLS = 2;
	/** Number of columns in label section of configSet */
	public static final int NUM_OF_CONFIGSET_LABEL_COLUMNS = 2;
	/** Total column count of mode and peripheral side by side */
	private static final int MODE_PERIPHERAL_COLUMNS = 10;
	/** Column count for peripheral when side by side with mode */
	private static final int PERIPHERAL_COLUMN_SPAN = 4;
	/** Column count for mode when side by side with peripheral */
	private static final int MODE_COLUMN_SPAN = 6;
	/** ID of the editor */
	public static final @NonNull String ID = "com.nxp.swtools.periphs.gui.view.componentsettings"; //$NON-NLS-1$
	/** Key for error decoration */
	public static final @NonNull String ERROR_DECORATION_KEY = "error_decoration"; //$NON-NLS-1$
	/** Minimal width of the editor in ems */
	private static final int MIN_WIDTH_EMS = SwToolsProduct.isUctProduct() ? 100 : 70; // FIXME Andreea v11 - Value over 70 does not fit FullHD display. Find another solution and then remove this condition.
	/** The "top-most" composite with scroll ability */
	protected ScrolledComposite scrolledComposite;
	/** Controller associated with the editor */
	final Controller controller = Controller.getInstance();
	/** Component editor input */
	ComponentSettingViewInput componentInput;
	/** {@link ComponentInstanceControl} associated with the editor */
	private @Nullable IChildControl componentSettingControl = null;
	/** Listeners which were registered and need to be unregistered during dispose */
	final @NonNull Collection<@NonNull IEventListener> registeredListeners = new ArrayList<>();
	/** Controls of all displayed config sets */
	final @NonNull Map<@NonNull IConfigSetConfig, ConfigSetControl> configSetControls = new HashMap<>();
	/** Combo used for configuration of component mode */
	private @Nullable InstantSearchList comboBoxMode;
	/** Resize listener instance */
	private @Nullable Listener resizeListener;
	/** Label for name textbox */
	private @Nullable Label labelName;
	/** Textbox for name */
	private @Nullable Text textBoxName;
	/** Label for mode combo */
	private @Nullable Label labelMode;
	/** Label for peripherals combo */
	private @Nullable Label labelPeripheral;
	/** Combobox for peripheral */
	private @Nullable InstantSearchList comboBoxPeripheral;
	/** Label for custom name field */
	private @Nullable Label labelCustomName;
	/** Button to enable/disable custom name */
	private @Nullable Button buttonCustomName;
	/** Stores validity of current name of instance */
	boolean nameIsValid = true;
	/** view selection provider for other plugins */
	protected GenericSelectionProvider viewSelectionProvider = new GenericSelectionProvider();
	/** Action which needs to be performed after the view becomes visible */
	@NonNull ActionRequired requiredAction = ActionRequired.NO_ACTION;
	/** Listener for validations changes */
	private @NonNull ValidationProblemListenerAdapter validationsListener = new ValidationProblemListenerAdapter() {
		@Override
		public void validationProblemsChanged(@NonNull Collection<@NonNull IValidationProblem> problems) {
			if ((contentComposite != null) && !contentComposite.isDisposed()) {
				refreshIfVisible(UpdateType.PROBLEM_DECORATION);
			}
		}
	};

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.ViewPart#init(org.eclipse.ui.IViewSite)
	 */
	@Override
	public void init(IViewSite site) throws PartInitException {
		setSite(site);
		String secondaryId = site.getSecondaryId();
		if (secondaryId == null) {
			return;
		}
		site.setSelectionProvider(viewSelectionProvider);
		Matcher matcher = SECONDARY_ID_GROUPS_PATTERN.matcher(secondaryId);
		String type, name, uuid;
		int occurrence = -1;
		type = name = uuid = UtilsText.EMPTY_STRING;
		if (matcher.matches()) {
			uuid = UtilsText.safeString(matcher.group(SECONDARY_ID_GROUP_UUID));
			type = UtilsText.safeString(matcher.group(SECONDARY_ID_GROUP_TYPE));
			name = UtilsText.safeString(matcher.group(SECONDARY_ID_GROUP_NAME));
			occurrence = Integer.parseInt(UtilsText.safeString(matcher.group(SECONDARY_ID_GROUP_OCCURRENCE)));
		}
		final int occurrenceFinal = occurrence;
		ComponentSettingViewInput input = new ComponentSettingViewInput(type, name, false);
		input.setUUID(uuid);
		this.componentInput = input;
		IChild inputSetting = getComponent();
		if (inputSetting instanceof IComponentInstanceConfig) {
			IComponentInstanceConfig instance = (IComponentInstanceConfig) inputSetting;
			componentInput.setUUID(instance.getUUID());
			if (instance.getComponent().isRegistersInitialization()) {
				site.getPage().showView(RegistersView.ID);
			}
		}
		getSiteSafe().getPage().addPartListener(new IPartListener() {
			/* (non-Javadoc)
			 * @see org.eclipse.ui.IPartListener#partOpened(org.eclipse.ui.IWorkbenchPart)
			 */
			@Override
			public void partOpened(IWorkbenchPart part) {
				if (part.equals(ComponentSettingView.this)) {
					IComponentConfig component = controller.getConfiguredComponent(getComponentType());
					IComponentInstanceConfig componentInstance;
					String uuid = componentInput.getUUID();
					if (UtilsText.isEmpty(uuid)) {
						componentInstance = controller.getComponentInstance(getComponentType(), getComponentName());
					} else {
						componentInstance = controller.getComponentInstance(uuid);
					}
					AStoragePeriphsComponent configToSave = null;
					// Use component instance if it is not null, otherwise use component (can be null)
					if (componentInstance != null) {
						configToSave = componentInstance.getStorageComponent();
						viewSelectionProvider.setSelection(new StructuredSelection(new ComponentInstanceWrapper(componentInstance, controllerWrapper)));
					} else if (component != null) {
						configToSave = component.getStorageComponent();
					} else {
						// probably residue from the previous run, will be closed later
						return;
					}
					setPartName((inputSetting == null) ? null : inputSetting.getUiName());
					ComponentSettingViewHelper.getInstance()
						.addEditor(Controller.getInstance().getFunctionalGroup().getStorageFuncGroup(), configToSave);
				}
			}

			/* (non-Javadoc)
			 * @see org.eclipse.ui.IPartListener#partDeactivated(org.eclipse.ui.IWorkbenchPart)
			 */
			@Override
			public void partDeactivated(IWorkbenchPart part) {
				// Do nothing
			}

			/* (non-Javadoc)
			 * @see org.eclipse.ui.IPartListener#partClosed(org.eclipse.ui.IWorkbenchPart)
			 */
			@Override
			public void partClosed(IWorkbenchPart part) {
				if (part.equals(ComponentSettingView.this)) {
					ComponentSettingViewInput inputOfClosedEditor = componentInput;
					if (inputOfClosedEditor != null) {
						ComponentSettingViewHelper.getInstance().removeEditor(inputOfClosedEditor,
								Controller.getInstance().getFunctionalGroup().getStorageFuncGroup().getUUID());
						ComponentSettingViewHelper.getInstance().removePermittedOccurrence(ComponentSettingViewHelper.createSecondaryId(inputOfClosedEditor),
								Integer.valueOf(occurrenceFinal));
					}
					getSiteSafe().getPage().removePartListener(this);
				}
			}

			/* (non-Javadoc)
			 * @see org.eclipse.ui.IPartListener#partBroughtToTop(org.eclipse.ui.IWorkbenchPart)
			 */
			@Override
			public void partBroughtToTop(IWorkbenchPart part) {
				handleViewFocusedEvent(part);
			}

			private void handleViewFocusedEvent(IWorkbenchPart part) {
				if (part.equals(ComponentSettingView.this)) {
					if (inputSetting instanceof IComponentInstanceConfig) {
						IComponentInstanceConfig instance = (IComponentInstanceConfig) inputSetting;
						if (instance.getComponent().isRegistersInitialization()) {
							RegistersModelSingleton.getInstance().setSelectedComponent(instance);
							controller.refreshRegistersView();
						}
					}
				}
			}

			/* (non-Javadoc)
			 * @see org.eclipse.ui.IPartListener#partActivated(org.eclipse.ui.IWorkbenchPart)
			 */
			@Override
			public void partActivated(IWorkbenchPart part) {
				handleViewFocusedEvent(part);
			}
		});
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#dispose()
	 */
	@Override
	public void dispose() {
		ValidationEngineFactory.removeListener(validationsListener);
		for (IEventListener listener : registeredListeners) {
			controller.removeListener(listener);
		}
		super.dispose();
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.view.ToolView#getDefaultCompositeSpecificImplementation(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected @NonNull Composite getDefaultCompositeSpecificImplementation(Composite basic) {
		return new ScrolledComposite(basic, SWT.V_SCROLL | SWT.H_SCROLL);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) {
		if (componentInput == null) {
			return;
		}
		ScrolledComposite scrolledCompositeLoc = (ScrolledComposite) createDefaultComposite(parent);
		scrolledComposite = scrolledCompositeLoc;
		SWTFactoryProxy.INSTANCE.setTestId(scrolledCompositeLoc, TestIDs.PERIPHS_COMPONENT_SETTING_VIEW_CONTENT);
		contentComposite = new Composite(scrolledCompositeLoc, SWT.NONE);
		GridLayoutComponents layout = new GridLayoutComponents(COMPONENT_COLS, false);
		layout.marginWidth = 8;
		layout.horizontalSpacing = 8;
		contentComposite.setLayout(layout);
		scrolledCompositeLoc.setContent(contentComposite);
		scrolledCompositeLoc.setExpandHorizontal(true);
		scrolledCompositeLoc.setExpandVertical(true);
		addPartHandlers();
		// create content of the view asynchronously as it is possible that the view becomes hidden meanwhile, postpone creation in that case
		parent.getDisplay().asyncExec(() -> {
			if (state == State.UNINITIALIZED) {
				// the view is not created yet
				state = State.READY_TO_CREATE;
				IWorkbenchPage page = getSiteSafe().getPage();
				IWorkbenchPart part = getSiteSafe().getPart();
				if ((page == null) || (part == null)) {
					LOGGER.fine("[TOOL] GUI: page or part is set to null"); //$NON-NLS-1$
					return;
				}
				if (page.isPartVisible(part)) {
					// the view is still visible, create its content
					recreate();
					contentComposite.forceFocus();
					state = State.CREATED_VISIBLE;
				}
			}
		});
		registerEventListener();
		ValidationEngineFactory.addListener(validationsListener);
		updateErrorIndicators();
	}

	/**
	 * Adds part listener handlers to this view
	 */
	protected void addPartHandlers() {
		getSiteSafe().getPage().addPartListener(new PartListener2Adapter() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.views.PartListener2Adapter#partVisible(org.eclipse.ui.IWorkbenchPartReference)
			 */
			@Override
			public void partVisible(IWorkbenchPartReference partRef) {
				if (getViewSiteNonNull().getPart() != partRef.getPart(false)) {
					return;
				}
				if (state == State.CREATED_HIDDEN) {
					// the view was hidden before, make it visible and refresh its content if required
					state = State.CREATED_VISIBLE;
					if (requiredAction == ActionRequired.RECREATE) {
						recreate();
					} else if (requiredAction == ActionRequired.REFRESH) {
						refreshSettings(UpdateType.NORMAL);
					}
					requiredAction = ActionRequired.NO_ACTION;
				} else if (state == State.READY_TO_CREATE) {
					// the view is ready to be created, create it
					state = State.CREATED_VISIBLE;
					recreate();
					requiredAction = ActionRequired.NO_ACTION;
				}
			}

			/* (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.views.PartListener2Adapter#partHidden(org.eclipse.ui.IWorkbenchPartReference)
			 */
			@Override
			public void partHidden(IWorkbenchPartReference partRef) {
				if (getViewSiteNonNull().getPart() == partRef.getPart(false) && (state == State.CREATED_VISIBLE)) {
					// the view was visible before, make it hidden and clear the required action flag
					state = State.CREATED_HIDDEN;
					requiredAction = ActionRequired.NO_ACTION;
				}
			}

			/* (non-Javadoc)
			 * @see com.nxp.swtools.common.ui.utils.views.PartListener2Adapter#partClosed(org.eclipse.ui.IWorkbenchPartReference)
			 */
			@Override
			public void partClosed(IWorkbenchPartReference partRef) {
				if (getViewSiteNonNull().getPart() == partRef.getPart(false)) {
					getSiteSafe().getPage().removePartListener(this);
					state = State.UNINITIALIZED;
					requiredAction = ActionRequired.NO_ACTION;
				}
			}
		});
	}

	/**
	 * @return nonnull IViewSite guarded by assert
	 */
	protected @NonNull IViewSite getViewSiteNonNull() {
		IViewSite site = getViewSite();
		assert site != null;
		return site;
	}

	/**
	 * Register listener responsible for closing the editor if it is not needed anymore or refreshing the content.
	 */
	private void registerEventListener() {
		// register close listener
		IEventListener changesListener = new IEventListener() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.utils.events.IEventListener#handle(com.nxp.swtools.utils.events.ToolEvent)
			 */
			@Override
			public void handle(ToolEvent event) {
				boolean shouldCloseView = false;
				boolean shouldRemoveViewFromOpened = false;
				String componentType = getComponentType();
				IComponentInstanceConfig componentSetting = Controller.getInstance().getComponentInstance(componentInput.getUUID());
				if (componentInput.isGlobalConfig()) {
					shouldCloseView = controller.getProfile().getConfiguredComponents().values().stream()
							.noneMatch(config -> config.getName().equals(componentType));
					// Global configuration does not exist -> remove it from opened views
					shouldRemoveViewFromOpened = shouldCloseView;
				} else {
					final String uuid = (componentSetting != null) ? componentSetting.getUUID() : UtilsText.EMPTY_STRING;
					shouldCloseView = controller.getFunctionalGroup().getInstances().values().stream()
							.noneMatch(instance -> instance.getUUID().equals(uuid));
					shouldRemoveViewFromOpened = controller.getProfile().getAllInstances().stream()
							.noneMatch(instance -> instance.getUUID().equals(uuid));
				}
				// Remove from opened views in helper when no configuration instance for this view exists
				if (shouldRemoveViewFromOpened){
					ComponentSettingViewInput componentInputReference = componentInput;
					if (componentInputReference != null) {
						ComponentSettingViewHelper.getInstance().removeEditor(componentInputReference,
								controller.getFunctionalGroup().getStorageFuncGroup().getUUID());
					}
				}
				// If no instance with required type and name exists then hide this view
				if (shouldCloseView) {
					Display display = Display.getCurrent();
					if ((display != null) && (!scrolledComposite.isDisposed())) {
						display.asyncExec(new Runnable() {
							@Override
							public void run() {
								IWorkbenchWindow workbenchWindow = getViewSiteNonNull().getWorkbenchWindow();
								if (workbenchWindow != null) {
									final IWorkbenchPage page = workbenchWindow.getActivePage();
									if (page != null) {
										page.hideView(ComponentSettingView.this);
									}
								}
							}
						});
					}
				}
				// refresh the content asynchronously (avoid concurrent modification exception)
				if (event.originator != ComponentSettingView.this) {
					if (state == State.CREATED_VISIBLE) {
						// the view is visible, recreate it
						Display display = Display.getCurrent();
						if (display != null) {
							display.asyncExec(() -> recreate());
						}
					} else if (state == State.CREATED_HIDDEN) {
						// the view is not visible, indicate that the view needs to be recreated
						requiredAction = ActionRequired.RECREATE;
					}
				}
				updateErrorIndicators();
				updateTooltips();
			}
		};
		controller.addListener(EventTypes.CHANGE, changesListener);
		registeredListeners.add(changesListener);
		// register setting value change listener
		IEventListener settingListener = new IEventListener() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.utils.events.IEventListener#handle(com.nxp.swtools.utils.events.ToolEvent)
			 */
			@Override
			public void handle(ToolEvent event) {
				refreshIfVisible(UpdateType.NORMAL);
			}
		};
		controller.addListener(EventTypes.SETTING_CHANGE, settingListener);
		registeredListeners.add(settingListener);
		// register initialization change listener
		IEventListener initializationListener = new IEventListener() {
			/* (non-Javadoc)
			 * @see com.nxp.swtools.utils.events.IEventListener#handle(com.nxp.swtools.utils.events.ToolEvent)
			 */
			@Override
			public void handle(ToolEvent event) {
				refreshIfVisible(UpdateType.INITIALIZATION);
			}
		};
		controller.addListener(EventTypes.INITIALIZATION, initializationListener);
		registeredListeners.add(initializationListener);
	}

	/**
	 * Refresh the View in case it is visible (update title in case the View is not visible).
	 * @param updateType type of the update
	 */
	protected void refreshIfVisible(@NonNull UpdateType updateType) {
		if (state == State.CREATED_HIDDEN) {
			if (requiredAction != ActionRequired.RECREATE) {
				// the refresh should be scheduled only if the recreate is not scheduled yet
				requiredAction = ActionRequired.REFRESH;
			}
			updateErrorIndicators();
		} else {
			refreshSettings(updateType);
		}
	}

	/**
	 * Update error indicator(s) of the editor's title given the current state.
	 */
	protected void updateErrorIndicators() {
		IChild component = getComponent();
		Image image = null;
		if (component != null) {
			int dependenciesLevel = ValidationHelper.getHighestSeverityComponentValidationProblemLevel(component);
			//FIXME TomasR v99 Create helper class for setting problems
			int settingsLevel = ErrorLevels.LEVEL_SUCCESS;
			if (component.getInfo() != null) {
				settingsLevel = ErrorLevels.LEVEL_INFORMATION;
			}
			if (component.getWarning() != null) {
				settingsLevel = ErrorLevels.LEVEL_WARNING;
			}
			if (component.getError() != null) {
				settingsLevel = ErrorLevels.LEVEL_ERROR;
			}
			final int problemLevel = (dependenciesLevel > settingsLevel) ? dependenciesLevel : settingsLevel;
			if (componentInput.isGlobalConfig()) {
				image = ToolView.getDecoratedImage(IToolsImages.IMAGE_EDITOR_COMPONENT_GLOBAL, problemLevel);
			} else {
				image = ToolView.getDecoratedImage(IToolsImages.IMAGE_EDITOR_COMPONENT_LOCAL, problemLevel);
			}
		}
		setTitleImage(image);
	}

	/**
	 * Update error decorators of the non-config set related editors (peripheral selection, mode selection, name selection).
	 */
	protected void updateErrorDecorators() {
		IComponentInstanceConfig compInstance = controller.getComponentInstance(componentInput.getUUID());
		if (compInstance != null) {
			Control[] editors = {textBoxName, comboBoxMode, comboBoxPeripheral};
			Supplier<?>[] suppliers = {() -> compInstance.getNotUniqueNameError(), null, () -> getPeripheralError(compInstance)};
			assert (editors.length == suppliers.length);
			for (int i = 0; i < editors.length; i++) {
				Control editor = editors[i];
				@SuppressWarnings("unchecked")
				Supplier<@Nullable String> supplier = (Supplier<@Nullable String>) suppliers[i];
				if ((editor != null) && (supplier != null)) {
					ControlDecorationUtils.updateControlDecoration(editor, supplier);
				}
			}
		}
	}

	/**
	 * Creates first row in the view.
	 * Contains Title, documentation link and enable/disable button if applicable.
	 * @param parent composite in which row should be created
	 */
	void createTitleRow(@NonNull Composite parent) {
		boolean isDocumentationPresent = false;
		ConfigurationComponentTypeId configCompTypeId = null;
		String titleString = UtilsText.EMPTY_STRING;
		String categoryId = UtilsText.EMPTY_STRING;
		IComponentInstanceConfig componentInstanceConfig = controller.getComponentInstance(componentInput.getUUID());
		IComponentConfig componentConfig = controller.getConfiguredComponent(getComponentType());
		int titleGridDataAdvancedColumns = 1; // default num columns

		Composite titleRow = new Composite(parent, SWT.NONE);
		titleRow.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, COMPONENT_COLS, 1));
		// Fix color on enable button in dark theme
		titleRow.setBackgroundMode(SWT.INHERIT_FORCE);
		// set layout to TITLE_ROW_COLS columns
		final GridLayoutComponents titleRowLayout = new GridLayoutComponents(TITLE_ROW_COLS, false);
		titleRowLayout.marginWidth = titleRowLayout.marginHeight = 0;
		titleRow.setLayout(titleRowLayout);

		// create title
		Composite titleComposite = new Composite(titleRow, SWT.NONE);
		titleComposite.setLayoutData(new GridDataComponents());
		RowLayout titleCompositeLayout = new RowLayout();
		titleCompositeLayout.marginHeight = 0;
		titleCompositeLayout.marginWidth = 0;
		titleComposite.setLayout(titleCompositeLayout);
		Label title = new Label(titleComposite, SWT.WRAP);

		IContext context = null;
		// get info for creation of link to documentation
		if (componentInput.isGlobalConfig() && (componentConfig != null)) {
			titleString = componentConfig.getComponent().getResolvedDescription(componentConfig.getExpressionContext());
			if (titleString == null) {
				titleString = componentConfig.getComponent().getLabel(componentConfig.getExpressionContext());
			}
			context = componentConfig.getExpressionContext();
			categoryId = componentConfig.getComponent().getCategory();
			configCompTypeId = componentConfig.getConfigCompTypeId();
			isDocumentationPresent = configCompTypeId.isDocumentationPresent();
		}
		if (!componentInput.isGlobalConfig() && (componentInstanceConfig != null)) {
			titleString = componentInstanceConfig.getComponent().getResolvedDescription(componentInstanceConfig.getExpressionContext());
			if (titleString == null) {
				titleString = componentInstanceConfig.getComponent().getLabel(componentInstanceConfig.getExpressionContext());
			}
			context = componentInstanceConfig.getExpressionContext();
			categoryId = componentInstanceConfig.getComponent().getCategory();
			configCompTypeId = componentInstanceConfig.getConfigCompTypeId();
			isDocumentationPresent = componentInstanceConfig.getConfigCompTypeId().isDocumentationPresent();
		}
		String categoryString = UtilsText.EMPTY_STRING;
		if (!UtilsText.isEmpty(categoryId) && (context != null)) {
			String categoryIdFinal = UtilsText.safeString(categoryId);
			IContext contextFinal = context;
			Categories categories = controllerWrapper.getController().getCategories();
			if (categories != null) {
				categoryString = categories.getCategoryLabel(categoryIdFinal, contextFinal);
			}
		}

		title.setLayoutData(new RowData());
		title.setText(titleString);
		FontFactory.scaleFontSize(title, 1.5);
		FontFactory.changeStyle(title, SWT.BOLD);
		Point titleSize = title.computeSize(SWT.DEFAULT, SWT.DEFAULT);
		if (!UtilsText.isEmpty(categoryString)) {
			Composite categoryComposite = new Composite(titleComposite, SWT.NONE);
			GridLayoutComponents categoryLayout = new GridLayoutComponents();
			categoryLayout.marginHeight = 0;
			categoryLayout.marginWidth = 0;
			categoryComposite.setLayout(categoryLayout);
			Label category = new Label(categoryComposite, SWT.NONE);
			category.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.BOTTOM, false, false));
			category.setText(UtilsText.LEFT_BRACKET + categoryString + UtilsText.RIGHT_BRACKET);
			SWTFactoryProxy.INSTANCE.setTestId(category, TestIDs.PERIPHS_CATEGORY_LABEL);
			FontFactory.changeStyle(category, SWT.ITALIC);
			Point categorySize = category.computeSize(SWT.DEFAULT, SWT.DEFAULT);
			categoryLayout.marginTop = titleSize.y - categorySize.y;
			categoryComposite.setLayoutData(new RowData(categorySize.x, titleSize.y));
		}

		// create documentation link
		if ((isDocumentationPresent) && (configCompTypeId != null)) {
			createLinkToDocumentation(titleRow, configCompTypeId);
		} else {
			titleGridDataAdvancedColumns++; // documentation button not needed -> take one more column
		}

		// if input is component instance config - create also enable/disable button
		if (!componentInput.isGlobalConfig() && (componentInstanceConfig != null)) {
			createLockButton(titleRow, componentInstanceConfig);
			createEnableButton(titleRow, componentInstanceConfig);
		} else {
			titleGridDataAdvancedColumns++; // no enable button -> take one more column
		}
		titleComposite.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.FILL, true, false, titleGridDataAdvancedColumns, 1));
	}

	/**
	 * Create GUI elements for the selected component.
	 */
	void createContent() {
		IChild inputSetting = getComponent();
		setPartName((inputSetting == null) ? null : inputSetting.getUiName());
		Composite contentCompositeLoc = contentComposite;
		if (contentCompositeLoc != null) {
			createTitleRow(contentCompositeLoc);
			if (componentInput.isGlobalConfig()) {
				createGlobalPart(contentCompositeLoc);
			} else {
				createInstancePart(contentCompositeLoc);
			}
		}
		updateErrorIndicators();
		updateErrorDecorators();
		updateTooltips();
		scrolledComposite.setMinWidth(GcUtils.getEmWidth(scrolledComposite, true) * MIN_WIDTH_EMS);
		scrolledComposite.setMinHeight(contentComposite.computeSize(scrolledComposite.getClientArea().width, SWT.DEFAULT).y);
		resizeListener = new Listener() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
			 */
			@Override
			public void handleEvent(Event event) {
				if (!getSiteSafe().getPage().isPartVisible(ComponentSettingView.this)) {
					return;
				}
				scrolledComposite.setMinHeight(contentComposite.computeSize(scrolledComposite.getClientArea().width, SWT.DEFAULT).y);
			}
		};
		contentComposite.addListener(SWT.Resize, resizeListener);
	}

	/**
	 * Creates part of view that controls global config set
	 * @param parent composite
	 * @deprecated - this part will be removed after new view for global config sets is approved
	 */
	@Deprecated
	private void createGlobalPart(@NonNull Composite parent) {
		IComponentConfig componentConfig = controller.getConfiguredComponent(getComponentType());
		if (componentConfig != null) {
			Label label = new Label(parent, SWT.NONE);
			label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, true, false, COMPONENT_COLS, 1));
			FontFactory.changeStyle(label, SWT.BOLD);
			label.setText(MessageFormat.format(UtilsText.safeString(Messages.get().ComponentEditor_GlobalConfigurationOf), componentConfig.getUiName()));
			if (!UtilsText.isEmpty(componentConfig.getDescription())) {
				label = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
				label.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, COMPONENT_COLS, 1));
				label = new Label(parent, SWT.WRAP);
				label.setLayoutData(new GridDataComponents(SWT.FILL, SWT.TOP, true, false, COMPONENT_COLS, 1));
				label.setText(UtilsText.safeString(componentConfig.getDescription()));
			}
			IConfigSetConfig configSetConfig = componentConfig.getGlobalConfigSet();
			if (configSetConfig != null) {
				updateConfigSetsContent(CollectionsUtils.asList(configSetConfig), parent);
			} else {
				label = new Label(parent, SWT.NONE);
				label.setText(Messages.get().ComponentSettingView_ComponentHasNoGlobalConfigSet);
				FontFactory.changeStyle(label, SWT.BOLD);
				FontFactory.scaleFontSize(label, 1.5);
			}
		}
	}

	/**
	 * Creates part of view that controls instance of components
	 * @param parent composite
	 */
	private void createInstancePart(@NonNull Composite parent) {
		IComponentInstanceConfig componentSetting = controller.getComponentInstance(componentInput.getUUID());
		if (componentSetting != null) {
			String comment = componentSetting.getComment();
			if (!comment.isEmpty()) {
				Label label = new Label(parent, SWT.NONE);
				labelName = label;
				label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false, 1, 1));
				label.setText(Messages.get().ComponentSettingView_CommentLabel);
				label = new Label(parent, SWT.NONE);
				labelName = label;
				label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false, COMPONENT_COLS - 1, 1));
				label.setText(UtilsText.cutOffAfter(comment, CUTOFF_COMMENT_LINES));
			}
			createNamePart(parent, componentSetting);
			MasterPeripheral masterPeripheral = componentSetting.getMode().getMasterPeripheral();
			boolean showModeControl = !componentSetting.isOptionSet(SettingOptions.UI_COMPONENT_MODE_HIDDEN);
			boolean showPeripheralControl = masterPeripheral != null;
			if (showModeControl && showPeripheralControl) {
				createModeAndPeripheralLine(parent, componentSetting, Objects.requireNonNull(masterPeripheral));
			} else {
				// Show only one of them
				if (showModeControl) {
					createModeControl(componentSetting, parent);
				}
				if (masterPeripheral != null) {
					createPeripheralControl(componentSetting, masterPeripheral, parent);
				}
			}
			// Link to component global config set
			String componentType = getComponentType();
			if (controller.hasGlobalConfigSet(componentType)) {
				createLinkToGlobalConfig(componentType);
			}
			createControlsOfInstance(parent, componentSetting);
			updateControls(UpdateType.NORMAL);
		}
	}

	/**
	 * @return type of the component for which this view was created
	 */
	private @NonNull String getComponentType() {
		String componentType = componentInput.getComponentType();
		if (UtilsText.isEmpty(componentType)) {
			IComponentInstanceConfig instance = controllerWrapper.getController().getComponentInstance(componentInput.getUUID());
			if (instance != null) {
				componentType = instance.getType();
			}
		}
		return componentType;
	}

	/**
	 * @return name of the component for which this view was created
	 */
	private String getComponentName() {
		String componentName = componentInput.getComponent();
		if (UtilsText.isEmpty(componentName)) {
			IComponentInstanceConfig instance = controllerWrapper.getController().getComponentInstance(componentInput.getUUID());
			if (instance != null) {
				componentName = instance.getName();
			}
		}
		return componentName;
	}

	/**
	 * Creates controls of instance
	 * @param parent composite
	 * @param instance
	 */
	private void createControlsOfInstance(@NonNull Composite parent, @NonNull IComponentInstanceConfig instance) {
		IChildControl componentSettingControlLoc = componentSettingControl;
		if ((componentSettingControlLoc == null) || (!componentSettingControlLoc.getChild().equals(instance)) || (componentSettingControlLoc.isDisposed())) {
			if (componentSettingControlLoc != null) {
				componentSettingControlLoc.dispose();
			}
			componentSettingControlLoc = ChildControlFactory.create(instance, controllerWrapper, null);
			componentSettingControl = componentSettingControlLoc;
			if (componentSettingControlLoc != null) {
				componentSettingControlLoc.setParentControl(null);
				componentSettingControlLoc.getControlOptions().labelHidden(true);
				componentSettingControlLoc.getControlOptions().borderHidden(true);
			}
		}
		if (componentSettingControlLoc != null) {
			componentSettingControlLoc.create(parent, COMPONENT_COLS);
		}
	}

	/**
	 * Creates part of view that controls mode and peripheral of instance
	 * @param parent composite
	 * @param instance
	 * @param masterPeripheral data
	 */
	private void createModeAndPeripheralLine(@NonNull Composite parent, @NonNull IComponentInstanceConfig instance, @NonNull MasterPeripheral masterPeripheral) {
		Composite modePeripheralComposite = new Composite(parent, SWT.NONE);
		GridLayoutComponents layout = new GridLayoutComponents(MODE_PERIPHERAL_COLUMNS, true);
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		modePeripheralComposite.setLayout(layout);
		modePeripheralComposite.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, false, 2, 1));
		Composite modeComposite = new Composite(modePeripheralComposite, SWT.NONE);
		GridLayoutComponents modeCompositeLayout = new GridLayoutComponents(COMPONENT_COLS, false);
		modeCompositeLayout.marginWidth = 0;
		modeCompositeLayout.marginHeight = 0;
		modeComposite.setLayout(modeCompositeLayout);
		modeComposite.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, false, MODE_COLUMN_SPAN, 1));
		createModeControl(instance, modeComposite);
		Composite peripheralComposite = new Composite(modePeripheralComposite, SWT.NONE);
		GridLayoutComponents peripheralCompositeLayout = new GridLayoutComponents(COMPONENT_COLS, false);
		peripheralCompositeLayout.marginWidth = 0;
		peripheralCompositeLayout.marginHeight = 0;
		peripheralComposite.setLayout(peripheralCompositeLayout);
		peripheralComposite.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, false, PERIPHERAL_COLUMN_SPAN, 1));
		createPeripheralControl(instance, masterPeripheral, peripheralComposite);
	}

	/**
	 * Creates part of view with name controls of instance
	 * @param parent composite
	 * @param instance
	 */
	private void createNamePart(@NonNull Composite parent, @NonNull IComponentInstanceConfig instance) {
		Composite nameComposite = new Composite(parent, SWT.NONE);
		GridLayoutComponents layout = new GridLayoutComponents(4, false);
		layout.marginWidth = 0;
		nameComposite.setLayout(layout);
		nameComposite.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, COMPONENT_COLS, 1));
		Label label = new Label(nameComposite, SWT.NONE);
		labelName = label;
		label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
		label.setText(Messages.get().ComponentEditor_Name);
		IChild component = getComponent();
		Text text = new Text(nameComposite, SWT.BORDER);
		if (instance.isCustomNameEnabled()) {
			text.setTextLimit(Limits.INPUT_LENGTH_C_COMPATIBLE);
		} else {
			text.setEnabled(false);
		}
		textBoxName = text;
		text.setText((component != null) ? component.getUiName() : UtilsText.EMPTY_STRING);
		// Handling
		TextBoxHelper.attachModifyListeners(text, (newName) -> {
			// change, use new name
			controller.runTransaction(() -> {
				// this must run as a transaction - listeners should be fired after the component input is updated,
				// the editor is closed otherwise because there is conflict in component name and name in component input
				if (controller.renameComponentInstance(instance, newName, ComponentSettingView.this, true)) {
					componentInput.setComponent(newName);
					setPartName(newName);
					updateControls(UpdateType.NORMAL);
				}
			});
		});
		TextBoxHelper.attachModifyErrorListener(() -> instance.getUiName(), text, (newName) -> {
			boolean isSameText = instance.getUiName().equalsIgnoreCase(newName);
			boolean isNameUsable = controller.isNameUsable(newName);
			String errorText = null;
			if (!isNameUsable) {
				errorText = Messages.get().ComponentEditor_NameNotUniqueError;
			}
			// if newName is the same as previous name -> check for uniqueness error
			// (can be caused by setting same prefix for fn groups which have comp. instances with same name)
			if (isSameText) {
				errorText = instance.getNotUniqueNameError();
			}
			final String error = errorText;
			if (errorText != null) {
				nameIsValid = false;
			} else {
				nameIsValid = true;
			}
			ControlDecorationUtils.updateControlDecoration(text, () -> error);
			updateTooltips();
			if (errorText != null) {
				return Status.INVALID;
			} else {
				return Status.OK;
			}
		});
		if (instance.getNotUniqueNameError() != null) {
			nameIsValid = false;
			text.setBackground(SwToolsColors.getColor(SwToolsColors.ERROR_BG));
			text.setForeground(SwToolsColors.getColor(SwToolsColors.ERROR_FG));
		}
		ControlDecorationUtils.createErrorDecoration(text, SWT.NONE, ToolsImages.getStatusDecoratorImg(ErrorLevels.LEVEL_ERROR));
		SWTFactoryProxy.INSTANCE.setTestId(text, TestIDs.PERIPHS_COMPONENT_NAME_EDITOR);
		text.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false));
		label = new Label(nameComposite, SWT.NONE);
		label.setText(Messages.get().ComponentSettingView_CustomName);
		ToolTipableImplementation tooltipCustomName = new ToolTipableImplementation();
		tooltipCustomName.setUiName(Messages.get().ComponentSettingView_CustomName);
		tooltipCustomName.setDescription(Messages.get().ComponentSettingView_CustomNameDescription);
		String toolTipText = ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(tooltipCustomName));
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(label, toolTipText);
		label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
		labelCustomName = label;
		Button button = new Button(nameComposite, SWT.CHECK);
		SWTFactoryProxy.INSTANCE.setTestId(button, TestIDs.PERIPHS_COMPONENT_NAME_CUSTOM_CHECKBOX);
		button.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
		button.setSelection(instance.isCustomNameEnabled());
		button.addSelectionListener(new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(SelectionEvent e) {
				controller.setInstanceToCustomNameMode(instance, button.getSelection(), ComponentSettingView.class);
			}
		});
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(button, toolTipText);
		buttonCustomName = button;
	}

	/**
	 * Creates peripheral combobox with label in given composite
	 * @param instance of component
	 * @param masterPeripheral master peripheral
	 * @param composite in which to create content
	 */
	private void createPeripheralControl(@NonNull IComponentInstanceConfig instance, @NonNull MasterPeripheral masterPeripheral,
			@NonNull Composite composite) {
		Label label;
		label = new Label(composite, SWT.NONE);
		labelPeripheral = label;
		label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
		label.setText(Messages.get().ComponentEditor_Peripheral);
		InstantSearchList combo = new InstantSearchList(composite, SWT.BORDER);
		comboBoxPeripheral = combo;
		SWTFactoryProxy.INSTANCE.setTestId(combo, TestIDs.PERIPHS_COMPONENT_PERIPHERAL_COMBO);
		GridDataComponents layoutData = new GridDataComponents(SWT.FILL, SWT.FILL, true, false);
		combo.setLayoutData(layoutData);
		IFunctionalGroup functionalGroup = instance.getChildContext().getFunctionalGroup();
		if (functionalGroup == null) {
			LOGGER.log(Level.SEVERE, "Instance {0} is not part of any functional group. Functional group is required here.", instance.getName()); //$NON-NLS-1$
			return;
		}
		@NonNull String[] items = masterPeripheral.getPeripherals().stream()
				.map(x -> functionalGroup.getAvailablePeripherals(x))
				.flatMap(x -> x.stream())
				.toArray(String[]::new);
		combo.setItems(items);
		IRegistersDatabaseAPI registersDb = controllerWrapper.getController().getMcu().getRegistersDb();
		if (registersDb != null) {
			for (String item : items) {
				StringBuilder builder = new StringBuilder(item);
				IRegParentPeripheralAPI peripheral = registersDb.getPeripheral(item);
				if (peripheral != null) {
					String fullName = peripheral.getFullName();
					if (fullName != null) {
						builder.append(UtilsText.SPACE).append(UtilsText.MINUS_STRING).append(UtilsText.SPACE).append(fullName);
					}
				}
				combo.addItemToolTip(item, (builder.length() == 0) ? null : builder.toString());
			}
		}

		ControlDecorationUtils.createErrorDecoration(combo, SWT.TOP | SWT.LEFT, ToolsImages.getStatusDecoratorImg(ErrorLevels.LEVEL_ERROR));

		String selectedPeripheral = instance.getPeripheral();
		if (selectedPeripheral != null) {
			combo.setSelection(selectedPeripheral);
		}
		combo.addSelectionListener(new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(SelectionEvent e) {
				IWorkbenchPartSite partSite = getSite();
				assert partSite != null;
				if (!partSite.getPage().isPartVisible(ComponentSettingView.this)) {
					return;
				}
				String peripheralToSet = UtilsText.safeString(((InstantSearchList)e.widget).getText());
				ProgressUtils.run((m) -> {
					String uuid = componentInput.getUUID();
					controller.setPeripheral(uuid, peripheralToSet, ComponentSettingView.this);
				});
				recreate();
			}
		});
		Color background = combo.getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND);
		if (instance.getMasterPeripheralError() != null) {
			background = SwToolsColors.getColor(SwToolsColors.ERROR_BG);
		}
		combo.setBackground(background);
	}

	/**
	 * Creates mode combobox with label in given composite
	 * @param instance of component
	 * @param composite in which to create content
	 */
	private void createModeControl(@NonNull IComponentInstanceConfig instance, @NonNull Composite composite) {
		Label label;
		label = new Label(composite, SWT.NONE);
		labelMode = label;
		label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
		label.setText(Messages.get().ComponentEditor_Mode);
		InstantSearchList modeComboLoc = new InstantSearchList(composite, SWT.BORDER);
		this.comboBoxMode = modeComboLoc;
		SWTFactoryProxy.INSTANCE.setTestId(modeComboLoc, TestIDs.PERIPHS_COMPONENT_MODE_COMBO);
		modeComboLoc.setLayoutData(new GridDataComponents(SWT.FILL, SWT.FILL, true, false));
		modeComboLoc.setItems(getAvailableModeLabels(instance));
		modeComboLoc.select(instance.getMode().getUIName(instance.getExpressionContext()));
		modeComboLoc.addSelectionListener(new SelectionAdapter() {
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(SelectionEvent e) {
				IWorkbenchPartSite site = getSite();
				assert site != null;
				if (!site.getPage().isPartVisible(ComponentSettingView.this)) {
					return;
				}
				String modeSelection = UtilsText.safeString(((InstantSearchList) e.widget).getText());
				String uuid = componentInput.getUUID();
				ProgressUtils.run(m -> controller.changeMode(uuid, modeSelection, ComponentSettingView.this));
				recreate();
			}
		});
	}

	/**
	 * Get array of available modes labels
	 * @param instance to get modes
	 * @return array of modes labels
	 */
	public @NonNull String[] getAvailableModeLabels(IComponentInstanceConfig instance) {
		return instance.getComponent().getScenarios().stream()
				.filter(y -> y.isAvailable(instance.getExpressionContext()))
				.filter(y -> {
					MasterPeripheral masterPeripheral = y.getMasterPeripheral();
					if (masterPeripheral == null) {
						return true;
					}
					for (String periph : masterPeripheral.getPeripherals()) {
						if (!controller.getMcu().getPeripherals(periph).isEmpty()) { // mcu contains at least one peripheral of the type from mode
							return true;
						}
					}
					return false;
				})
				.map(y -> y.getUIName(instance.getExpressionContext()))
				.toArray(String[]::new);
	}

	/**
	 * Returns error of this instance
	 * @param config of the component instance
	 * @return Error message
	 */
	private static @Nullable String getPeripheralError(@NonNull IComponentInstanceConfig config) {
		return config.getMasterPeripheralError();
	}

	/**
	 * Add link to component global config set
	 * @param componentType type of component that is being linked to
	 */
	private void createLinkToGlobalConfig(@NonNull String componentType) {
		// draw line
		Label label = new Label(contentComposite, SWT.SEPARATOR | SWT.HORIZONTAL);
		label.setLayoutData(new GridDataComponents(SWT.FILL, SWT.CENTER, true, false, COMPONENT_COLS, 1));

		// add space after line - consistent with configSetControl
		label = new Label(contentComposite, SWT.NONE);
		label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.TOP, false, false, COMPONENT_COLS, 1));

		// Title
		label = new Label(contentComposite, SWT.NONE);
		label.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.TOP, false, false, 1, 1));
		label.setText(Messages.get().ComponentEditor_ComponentGlobalSettings);
		FontFactory.changeStyle(label, SWT.BOLD);

		// link to global config set editor
		Link link = new Link(contentComposite, SWT.NONE);
		link.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false, COMPONENT_COLS - 1, 1));
		link.setText("<A>" + componentType + "</A>"); //$NON-NLS-1$ //$NON-NLS-2$
		SWTFactoryProxy.INSTANCE.setTestId(link, TestIDs.PERIPHS_COMPONENT_GLOBAL_CONFIG_SET_LINK);

		link.addListener(SWT.Selection, new Listener() {
			@Override
			public void handleEvent(Event event) {
				// open component global settings editor
				GlobalComponentSettingView.open(getSiteSafe().getPage(), componentType, null);
			}
		});
	}

	/**
	 * Creates button into parent composite that opens documentation view with documentation for given ConfigurationComponentTypeId.
	 * @param parent composite into which button should be created
	 * @param confCompTypeId ConfigurationComponentTypeId that contains documentation to show
	 */
	private static void createLinkToDocumentation(@NonNull Composite parent, @NonNull ConfigurationComponentTypeId confCompTypeId) {
		final Button btn = new Button(parent, SWT.NONE);
		final Image img = ToolsImages.getImage(IToolsImages.ICON_DOCUMENTATION);
		if (img == null) {
			assert false : "image not found: " + IToolsImages.ICON_DOCUMENTATION; //$NON-NLS-1$
		} else {
			btn.setImage(img);
		}
		SWTFactoryProxy.INSTANCE.setTestId(btn, TestIDs.PERIPHS_COMPONENT_SETTING_VIEW_DOCUMENTATION);
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(btn, UtilsText.safeString(Messages.get().ComponentSettingView_Documentation));
		btn.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
		btn.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				IViewPart viewPart = ComponentSettingViewHelper.getViewPart();
				if (viewPart != null) {
					IViewSite viewSite = viewPart.getViewSite();
					if (viewSite != null) {
						DocumentationView.open(viewSite, confCompTypeId.getTypeId(), true);
					}
				}
			}
		});
	}

	/**
	 * Creates button for enabling/disabling component instance.
	 * @param parent composite into which button should be created
	 * @param componentInstanceConfig IComponentInstanceConfig that state is being changed by button
	 */
	private void createEnableButton(@NonNull Composite parent, @NonNull IComponentInstanceConfig componentInstanceConfig) {
		Runnable action = new Runnable() {
			@Override
			public void run() {
				ProgressUtils.run(m -> controller.setComponentInstancesEnabled(CollectionsUtils.asList(componentInstanceConfig), !componentInstanceConfig.isEnabled(), this));
				recreate();
			}
		};
		ToggleButton toggleButton = new ToggleButton(parent, componentInstanceConfig.isEnabled(), TestIDs.PERIPHS_ENABLE_INSTANCE_BUTTON + getComponentName(), action);
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(toggleButton, Messages.get().ComponentSettingView_EnableDisableComponentInstance);
		SWTFactoryProxy.INSTANCE.setTestId(toggleButton, TestIDs.PERIPHS_ENABLE_INSTANCE_BUTTON + getComponentName());
		toggleButton.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
		ToolView.disableGlobalControlOfEnabledFlag(toggleButton);
	}

	/**
	 * Creates button for locking/unlocking editing of component instance
	 * @param parent composite into which button should be created
	 * @param componentInstanceConfig IComponentInstanceConfig of which the editing lock state is being changed by button
	 */
	private void createLockButton(@NonNull Composite parent, @NonNull IComponentInstanceConfig componentInstanceConfig) {
		Runnable action = new Runnable() {
			@Override
			public void run() {
				ProgressUtils.run(m -> controller.setEditingLockOfInstance(componentInstanceConfig, !componentInstanceConfig.isEditingLocked()));
			}
		};
		Button button = new Button(parent, SWT.TOGGLE);
		button.setImage(ToolsImages.getImage(componentInstanceConfig.isEditingLocked() ? IToolsImages.ICON_LOCKED : IToolsImages.ICON_UNLOCKED));
		button.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(@NonNull SelectionEvent e) {
				action.run();
			}
		});
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(button, componentInstanceConfig.isEditingLocked() ? Messages.get().ComponentSettingView_UnlockEditingOfComponentInstance
				: Messages.get().ComponentSettingView_LockEditingOfComponentInstance);
		SWTFactoryProxy.INSTANCE.setTestId(button, TestIDs.PERIPHS_LOCK_EDITING_BUTTON);
		button.setLayoutData(new GridDataComponents(SWT.LEFT, SWT.CENTER, false, false));
	}

	/**
	 * Create controls for given config sets (or update if already exist).
	 * @param configSets for which to create GUI
	 * @param composite in which to create the GUI
	 */
	private void updateConfigSetsContent(@NonNull List<IConfigSetConfig> configSets, @NonNull Composite composite) {
		// drop the unused controls
		configSetControls.keySet().retainAll(configSets);
		for (IConfigSetConfig configSet : configSets) {
			ConfigSetControl configSetControl = configSetControls.get(configSet);
			if (configSetControl == null) { // control for the config set does not exist, create one
				configSetControl = (ConfigSetControl) ChildControlFactory.create(configSet, PeriphControllerWrapper.getInstance(), null);
			}
			if (configSetControl == null) {
				LOGGER.log(Level.SEVERE, "[TOOL] Attempt to create control failed for config set: {0}", configSet); //$NON-NLS-1$
				continue;
			}
			// create GUI
			configSetControl.create(composite, CONFIG_SET_COLS);
			configSetControl.update(UpdateType.NORMAL);
			configSetControls.put(configSet, configSetControl);
		}
	}

	/**
	 * Refresh content of the editor.
	 */
	void recreate() {
		if (contentComposite.isDisposed()) {
			return;
		}
		contentComposite.setRedraw(false);
		try {
			if (resizeListener != null) {
				contentComposite.removeListener(SWT.Resize, resizeListener);
			}
			if (componentSettingControl != null) {
				componentSettingControl.dispose();
			}
			for (ConfigSetControl configSetControl : configSetControls.values()) {
				configSetControl.dispose();
			}
			for (Control ctrl : contentComposite.getChildren()) {
				ctrl.dispose();
			}
			createContent();
		} catch (ExpressionException e) {
			e.log();
		} finally {
			contentComposite.setRedraw(true);
		}
		contentComposite.layout();
	}

	/**
	 * Open component editor.
	 * @param viewSite with binding to an editor
	 * @param componentType type of the component to edit
	 * @param componentName name of component to edit (component type in case global is set to {@code true})
	 * @param global whether editor for global configSet or for component instance should be opened
	 * @param activate whether to activate the view after creating
	 * @return {@code true} if an editor was successfully opened, {@code false} otherwise
	 */
	public static boolean open(@NonNull IViewSite viewSite, @NonNull String componentType, @NonNull String componentName,
			boolean global, boolean activate) {
		IWorkbenchPage iWorkbenchPage = viewSite.getPage();
		if (iWorkbenchPage != null) {
			return open(iWorkbenchPage, componentType, componentName, global, activate, null);
		}
		return false;
	}

	/**
	 * Open component editor.
	 * @param viewSite with binding to an editor
	 * @param uuid of the instance
	 * @param global whether editor for global configSet or for component instance should be opened
	 * @param activate whether to activate the view after creating
	 * @return {@code true} if an editor was successfully opened, {@code false} otherwise
	 */
	public static boolean open(@NonNull IViewSite viewSite, @NonNull String uuid, boolean activate) {
		IWorkbenchPage iWorkbenchPage = viewSite.getPage();
		if (iWorkbenchPage != null) {
			return open(iWorkbenchPage, uuid, activate, null);
		}
		return false;
	}

	/**
	 * Open component editor and show the problematic setting, if any
	 * @param workbenchPage with binding to an editor
	 * @param componentType type of the component to edit
	 * @param componentName name of component to edit (component type in case global is set to {@code true})
	 * @param global whether editor for global configSet or for component instance should be opened
	 * @param activate whether to activate the view after creating
	 * @param problematicChild of a setting, if any, <code>null</code> otherwise
	 * @return {@code true} if an editor was successfully opened, {@code false} otherwise
	 */
	public static boolean open(@Nullable IWorkbenchPage workbenchPage, @NonNull String componentType,
			@NonNull String componentName, boolean global, boolean activate, @Nullable IChild problematicChild) {
		if (workbenchPage != null) {
			// Reopen views that have their secondary ID based on old config name
			ComponentSettingViewHelper.getInstance().reopenInvalidViews();
			try {
				if (global) {
					componentName = UtilsText.EMPTY_STRING;
				}
				ComponentSettingViewInput input = new ComponentSettingViewInput(componentType, componentName, false);
				String identificationBase = ComponentSettingViewHelper.createSecondaryId(input);
				openView(workbenchPage, identificationBase, activate, problematicChild);
				return true;
			} catch (@SuppressWarnings("unused") PartInitException e) {
				return false;
			}
		}
		return false;
	}

	/**
	 * Open view - common part
	 * @param workbenchPage page of the workbench
	 * @param identificationBase to which the occurrence will be added
	 * @param activate {@code true} when the view should be focused after it is opened
	 * @param problematicChild which will be focused
	 * @throws PartInitException
	 */
	private static void openView(@NonNull IWorkbenchPage workbenchPage, @NonNull String identificationBase, boolean activate, @Nullable IChild problematicChild) throws PartInitException {
		Integer lastOpenOccurrence = Integer.valueOf(-1);
		Set<@NonNull Integer> permittedOccurrences = ComponentSettingViewHelper.getInstance().getPermittedOccurrences(identificationBase);
		for (Integer i : permittedOccurrences) {
			String identification = identificationBase + SECONDARY_ID_IDENTIFICATION_OCCURENCE_SEPARATOR + i.toString();
			boolean viewPresent = Arrays.asList(workbenchPage.getViewReferences()).stream()
				.filter(x -> ID.equals(x.getId()))
				.anyMatch(x -> identification.equals(x.getSecondaryId()));
			if (!viewPresent) {
				IViewPart view = workbenchPage.showView(ComponentSettingView.ID, identification,
						activate ? IWorkbenchPage.VIEW_ACTIVATE : IWorkbenchPage.VIEW_VISIBLE);
				if ((problematicChild != null) && (view != null)) {
					focusOnChild(view, problematicChild);
				}
				return;
			} else {
				lastOpenOccurrence = i;
			}
		}
		if (lastOpenOccurrence.intValue() != -1) {
			IViewPart view = workbenchPage.showView(ComponentSettingView.ID, identificationBase + SECONDARY_ID_IDENTIFICATION_OCCURENCE_SEPARATOR + lastOpenOccurrence.toString(),
					activate ? IWorkbenchPage.VIEW_ACTIVATE : IWorkbenchPage.VIEW_VISIBLE);
			if ((problematicChild != null) && (view != null)) {
				focusOnChild(view, problematicChild);
			}
		}
	}

	/**
	 * Open component editor and show the problematic setting, if any
	 * @param workbenchPage with binding to an editor
	 * @param uuid of instance which should be opened
	 * @param global whether editor for global configSet or for component instance should be opened
	 * @param activate whether to activate the view after creating
	 * @param problematicChild of a setting, if any, <code>null</code> otherwise
	 * @return {@code true} if an editor was successfully opened, {@code false} otherwise
	 */
	public static boolean open(@Nullable IWorkbenchPage workbenchPage, @NonNull String uuid, boolean activate, @Nullable IChild problematicChild) {
		if (workbenchPage != null) {
			// Reopen views that have their secondary ID based on old config name
			ComponentSettingViewHelper.getInstance().reopenInvalidViews();
			try {
				ComponentSettingViewInput input = new ComponentSettingViewInput(UtilsText.EMPTY_STRING, UtilsText.EMPTY_STRING, false);
				input.setUUID(uuid);
				openView(workbenchPage, ComponentSettingViewHelper.createSecondaryId(input), activate, problematicChild);
				return true;
			} catch (@SuppressWarnings("unused") PartInitException e) {
				return false;
			}
		}
		return false;
	}

	/**
	 * Scrolls to the required setting and focuses on it, if possible.
	 * @param view instance of the component where the required child is defined
	 * @param child to be found
	 */
	public static void focusOnChild(@NonNull IViewPart view, @NonNull IChild child) {
		if (view instanceof ComponentSettingView) {
			ComponentSettingView settingView = (ComponentSettingView) view;
			// focus on the required view if this is not
			if (!settingView.isViewFocused()) {
				Objects.requireNonNull(settingView.getToolMainControl()).setFocus();
			}
			// run the code when the UI items are visible
			settingView.contentComposite.getDisplay().asyncExec(() -> {
				IComponentInstanceConfig compInstance = settingView.controller.getComponentInstance(settingView.componentInput.getUUID());
				if (compInstance != null) {
					String childId = child.getId();
					String compInstanceName = compInstance.getName();
					IChildControl instanceControl = settingView.componentSettingControl;
					if (instanceControl != null) {
						Composite composite = instanceControl.getContent();
						if (composite != null) {
							ScrolledComposite scrolledComposite = ScrolledCompositeHelper.findScrolledComposite(composite);
							if (scrolledComposite != null) {
								// open all the views on the path to the child
								if (SwToolsProduct.isUctProduct()) { // FIXME TomasR, Ionut T. v13 maintenance - include changes from UCT to new function, try new function in UCT, remove condition
									ComponentSettingViewHelper.selectChildrenOnPath(instanceControl, child.getId(), null, compInstance.isEditingLocked() ? UpdateType.FORCE_DISABLE : UpdateType.NORMAL);
								} else {
									ComponentSettingViewHelper.selectChildrenOnPath(instanceControl, child.getId(), compInstance.isEditingLocked() ? UpdateType.FORCE_DISABLE : UpdateType.NORMAL);
								}
								// if the child belongs to one of the component's fields,
								// not to one of it's children, scroll to the top
								if (childId.equals(compInstanceName)) {
									ScrolledCompositeHelper.scrollByNumberOfLines(composite, 1, -scrolledComposite.getOrigin().y);
								} else {
									String childTestID = TestIDs.PERIPHS_SETTING_CONTROL + childId;
									Control control = ScrolledCompositeHelper.getControlWithTestId(composite, childTestID);
									// the required setting has a control and an attached test ID
									if (control != null) {
										if (KEPreferences.isAnimationsEnabled()) {
											ScrolledCompositeHelper.animateScrollToControlWithId(composite, childTestID);
										} else {
											ScrolledCompositeHelper.scrollToControlWithId(composite, childTestID);
										}
										control.setFocus();
									} else {
										// the required setting is in an array
										Integer offset = getOffsetToArrayRow(instanceControl, childId);
										if (offset != null) {
											// scroll to array row after the selection of
											// children on the path to the child has finished and
											// the items are visible
											composite.getDisplay().asyncExec(() -> {
												ScrolledCompositeHelper.scrollByNumberOfLines(composite, 1, offset.intValue());
											});
										}
									}
								}
							}
						}
					}
				}
			});
		}
	}

	/**
	 * Finds the scroll offset to a problematic row in an array table
	 * @param childControl the starting point on the path to the problem
	 * @param childId the ID of the problem
	 * @return integer containing the scroll offset to a problematic row in an array table
	 */
	private static Integer getOffsetToArrayRow(@NonNull IChildControl childControl, @NonNull String childId) {
		Integer offset = null;
		String currentChildID = childControl.getChild().getId();
		// the problematic setting is found
		if (childId.equals(currentChildID)) {
			// find the number of the row in the array table and return it
			return getArrayRowNumber(childControl.getChild());
		} else {
			// keep searching for the problematic row and visit the children
			if (childControl instanceof ChildProvidableControlBase) {
				ChildProvidableControlBase childProvidableControlBase = (ChildProvidableControlBase) childControl;
				List<@NonNull IChildControl> children = childProvidableControlBase.getChildren();
				for (IChildControl child : children) {
					offset = getOffsetToArrayRow(child, childId);
					// the row number has been found
					if (offset != null) {
						// get the control of the array and calculate the offset
						// from the current scrollbar's position to the problematic row
						if (childControl instanceof ArrayControl) {
							ArrayControl arrayControl = (ArrayControl) childControl;
							return arrayControl.getOffsetToRow(offset.intValue());
						}
						return offset;
					}
				}
			}
		}
		return offset;
	}

	/**
	 * Finds the row number of a child in the array it belongs to.
	 * @param child to find the row number of
	 * @return the row number if any, or <code>null</code> otherwise
	 */
	private static Integer getArrayRowNumber(IChild child) {
		if (child != null) {
			IChildProvidable parent = child.getChildContext().getParent();
			if ((parent != null) && (parent instanceof ArrayConfig)) {
				return new Integer(parent.getChildren().indexOf(child));
			}
			return getArrayRowNumber(parent);
		}
		return null;
	}

	/*
	 * (non-Javadoc)
	 * @see com.nxp.swtools.utils.view.ToolView#refreshStatus()
	 */
	@Override
	protected void refreshStatus() {
		refreshStatusBar();
	}

	/**
	 * Refresh settings, errors indicators, available modes.
	 * @param updateType type of the update
	 */
	void refreshSettings(@NonNull UpdateType updateType) {
		// update error indicator
		updateErrorIndicators();
		// update controls
		updateControls(updateType);
		// update error decorations of the non-config set related editors
		updateErrorDecorators();
		updateTooltips();
		if (updateType != UpdateType.PROBLEM_DECORATION) {
			// update available modes
			IComponentInstanceConfig componentSetting = controller.getComponentInstance(getComponentType(), getComponentName());
			InstantSearchList combo = comboBoxMode;
			if ((componentSetting != null) && (combo != null) && !combo.isDisposed()) {
				combo.setItems(getAvailableModeLabels(componentSetting));
				combo.select(componentSetting.getMode().getUIName(componentSetting.getExpressionContext()));
			}
			// update dimensions of scrollable pane
			scrolledComposite.setMinHeight(contentComposite.computeSize(scrolledComposite.getClientArea().width, SWT.DEFAULT).y);
		}
	}

	/**
	 * Update all controls in config sets.
	 * @param updateType type of the update
	 */
	void updateControls(@NonNull UpdateType updateType) {
		contentComposite.setRedraw(false);
		Profiler profiler = Profiler.getInstance(SwToolsProduct.PRODUCT_ID_PERIPHERALS_TOOL);
		BigInteger index = profiler.start(ComponentSettingView.class, !profiler.isEnabled() ? UtilsText.EMPTY_STRING :
				"Component settings view update of " + getComponentName()); //$NON-NLS-1$
		final IChild component = getComponent();
		if (component != null) {
			updateType = ((IComponentInstanceConfig) component).isEditingLocked() ? UpdateType.FORCE_DISABLE : updateType;
		}
		final UpdateType finalUpdateType = updateType;
		try {
			if (componentSettingControl != null) {
				componentSettingControl.update(finalUpdateType);
			}
			configSetControls.values().forEach(x -> x.update(finalUpdateType));
			if (finalUpdateType != UpdateType.PROBLEM_DECORATION) {
				final boolean enabled = (component != null) && component.isEnabled() && !((IComponentInstanceConfig) component).isEditingLocked();
				@Nullable Control[] controls = new @Nullable Control[] {comboBoxMode, comboBoxPeripheral, buttonCustomName, labelName, labelMode, labelPeripheral, labelCustomName};
				for (Control control : controls) {
					if ((control != null) && !control.isDisposed()) {
						control.setEnabled(enabled);
					}
				}
				Text textBoxNameLoc = textBoxName;
				if ((textBoxNameLoc != null) && (component != null)) {
					boolean customNameEnabled = ((IComponentInstanceConfig) component).isCustomNameEnabled();
					textBoxNameLoc.setEnabled(customNameEnabled && enabled);
				}
				updateTooltips();
			}
		} finally {
			contentComposite.setRedraw(true);
			profiler.stop(index, ComponentSettingView.class, null);
		}
	}

	/**
	 * Updates texts of tooltips and sets them on labels and editors
	 */
	void updateTooltips() {
		ToolTipableImplementation tooltipName = new ToolTipableImplementation();
		ToolTipableImplementation tooltipMode = new ToolTipableImplementation();
		ToolTipableImplementation tooltipPeripheral = new ToolTipableImplementation();
		// Update texts
		IComponentInstanceConfig componentSetting = controller.getComponentInstance(componentInput.getUUID());
		if (componentSetting == null) {
			return;
		}
		// Name
		Text textbox = textBoxName;
		if (textbox != null) {
			if (!textbox.isDisposed()) {
				String nameValue = UtilsText.safeString(textbox.getText());
				String error = getNameErrorMessageTooltip(componentSetting, nameValue);
				tooltipName.setError(error);
				tooltipName.setValueName(nameValue);
			}
			tooltipName.setUiName(Messages.get().ComponentEditor_Name);
			tooltipName.setDescription(Messages.get().ComponentEditor_NameDescription);
		}
		// Mode
		InstantSearchList combobox = comboBoxMode;
		if (combobox != null) {
			if (!combobox.isDisposed()) {
				String markDownToHtml = MarkDownSupport.markDownToHtml(combobox.getText() + UtilsText.SPACE + UtilsText.EN_DASH + UtilsText.SPACE +
						componentSetting.getMode().getDescription(), null);
				tooltipMode.setValueName(markDownToHtml);
			}
			tooltipMode.setUiName(Messages.get().ComponentEditor_Mode);
			tooltipMode.setDescription(Messages.get().ComponentEditor_ModeDescription);
		}
		// Peripheral
		combobox = comboBoxPeripheral;
		if ((combobox != null) && !combobox.isDisposed()) {
			String peripheral = combobox.getText();
			if (!combobox.isDisposed()) {
				tooltipPeripheral.setValueName(peripheral);
				IRegistersDatabaseAPI registersDb = controllerWrapper.getController().getMcu().getRegistersDb();
				if (registersDb != null) {
					IRegParentPeripheralAPI peripheralDb = registersDb.getPeripheral(peripheral);
					if (peripheralDb != null) {
						tooltipPeripheral.setValueDescription(peripheralDb.getFullName());
					}
				}
			}
			tooltipPeripheral.setUiName(Messages.get().ComponentEditor_Peripheral);
			tooltipPeripheral.setDescription(Messages.get().ComponentEditor_PeripheralDescription);
			tooltipPeripheral.setError(getPeripheralError(componentSetting));
		}
		// Update controls with tooltips
		@NonNull ToolTipableImplementation tooltips @NonNull [] = new @NonNull ToolTipableImplementation[]
				{tooltipName, tooltipMode, tooltipPeripheral};
		Label[] labels = {labelName, labelMode, labelPeripheral};
		Control[] editors = {textBoxName, comboBoxMode, comboBoxPeripheral};
		// Iterate over tooltips, labels and controls
		for (int index = 0; index < tooltips.length; index++) {
			@NonNull ToolTipableImplementation tooltip = tooltips[index];
			Label label = labels[index];
			Control editor = editors[index];
			String toolTipText = ToolTipableFormatter.getToolTipText(new ToolTipableMarkdownDescriptionDecorator(tooltip));
			Label nonNullLabel = label;
			if ((nonNullLabel != null) && (!nonNullLabel.isDisposed())) {
				SWTFactoryProxy.INSTANCE.setHtmlTooltip(nonNullLabel, toolTipText);
			}
			Control nonNullEditor = editor;
			if ((nonNullEditor != null) && (!nonNullEditor.isDisposed())) {
				SWTFactoryProxy.INSTANCE.setHtmlTooltip(nonNullEditor, toolTipText);
			}
		}
	}

	/**
	 * Returns error message when some errors are found and {@code null} when no errors are found
	 * @param instance to get functional group and profile from
	 * @param newName new name of instance
	 * @return {@code null} if there is no conflict and error message when there is a name conflict
	 */
	private @Nullable String getNameErrorMessageTooltip(@NonNull IComponentInstanceConfig instance, @NonNull String newName) {
		boolean nameIsIdentifier = APeriphController.isNameValid(newName);
		String error = null;
		if (!nameIsValid) {
			if (nameIsIdentifier) {
				error = instance.getNameConflictErrorInternal(instance, newName);
			} else {
				error = Messages.get().ComponentEditor_NameNotIdentifierError;
			}
		} else {
			String notUniqueNameError = instance.getNotUniqueNameError();
			if (notUniqueNameError != null) {
				error = notUniqueNameError;
			}
		}
		return error;
	}

	/**
	 * Get component from the view's input.
	 * @return component from the view's input or {@code null} in case the specified component does not exist
	 */
	private @Nullable IChild getComponent() {
		ComponentSettingViewInput componentInputLoc = componentInput;
		if (componentInputLoc != null) {
			if (componentInputLoc.isGlobalConfig()) {
				return controller.getConfiguredComponent(componentInputLoc.getComponentType());
			}
			if(!componentInputLoc.getUUID().isEmpty()) {
				// Use reference by UUID if it is already present
				return controller.getComponentInstance(componentInputLoc.getUUID());
			}
			return controller.getComponentInstance(componentInputLoc.getComponentType(), componentInputLoc.getComponent());
		}
		return null;
	}
}
