/**
 * Copyright 2022-2023 NXP
 */
package com.nxp.swtools.periphs.gui.view.componentsettings;

import java.util.List;
import java.util.logging.Logger;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewSite;

import com.nxp.swtools.common.ui.utils.swt.SWTFactoryProxy;
import com.nxp.swtools.common.ui.utils.swt.widgets.InstantSearchList;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.lang.CollectionsUtils;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.configuration.properties.SWToolsProperties;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.derivative.swt.GridDataComponents;
import com.nxp.swtools.derivative.swt.GridLayoutComponents;
import com.nxp.swtools.periphs.gui.controller.IControllerWrapper;
import com.nxp.swtools.periphs.gui.view.AddComponentDialog;
import com.nxp.swtools.periphs.model.data.ConfigurationComponent;
import com.nxp.swtools.resourcetables.model.config.IChildProvidable;
import com.nxp.swtools.resourcetables.model.config.ScalarConfig;
import com.nxp.swtools.resourcetables.model.data.ConfigurationComponentTypeId;
import com.nxp.swtools.utils.TestIDs;
import com.nxp.swtools.utils.progress.ProgressUtils;
import com.nxp.swtools.utils.resources.IToolsImages;
import com.nxp.swtools.utils.resources.ToolsImages;

/**
 * Class representing control of a combo-box setting with FEATURE_COMPONENT_ADD option
 * @author Ionut-Cristian Tudose - nxf64017
 */
public class ScalarComboAddComponentControl extends ScalarComboControl {
	/** Logger of the class */
	private static final @NonNull Logger LOGGER = LogManager.getLogger(ScalarComboAddComponentControl.class);
	/** Number of columns used when FEATURE_COMPONENT_ADD is defined for a setting */
	private static final int FEATURE_COMPONENT_ADD_NUM_COLUMNS = 2;

	/** Add button */
	private Button addButton;

	/**
	 * @param child to create control for
	 * @param controlOptions for this control
	 * @param controllerWrapper containing the generic controller
	 */
	public ScalarComboAddComponentControl(@NonNull ScalarConfig child, @NonNull ControlOptions controlOptions, @NonNull IControllerWrapper controllerWrapper) {
		super(child, controlOptions, controllerWrapper);
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarComboControl#setEnabledStateToMainControl(boolean)
	 */
	@Override
	protected void setEnabledStateToMainControl(boolean enabled) {
		InstantSearchList comboLoc = combo;
		if (comboLoc != null && comboLoc.getEnabled() != enabled) {
			comboLoc.setEnabled(enabled);
		}
		Button addButtonLoc = addButton;
		if (addButtonLoc != null && addButtonLoc.getEnabled() != enabled) {
			addButtonLoc.setEnabled(enabled && !controllerWrapper.getController().getComponentTypeIdsForComponentAddFeature(getChild()).isEmpty());
		}
		Control mainControlLoc = mainControl;
		if (mainControlLoc != null && !mainControlLoc.getEnabled()) {
			mainControlLoc.setEnabled(true); // Enable composite to allow tooltips to be shown
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarComboControl#createComposite(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected Composite createComposite(@NonNull Composite composite) {
		Composite contentComposite = new Composite(composite, SWT.NONE);
		GridLayoutComponents layout = new GridLayoutComponents(FEATURE_COMPONENT_ADD_NUM_COLUMNS, false);
		layout.marginWidth = layout.marginHeight = 0;
		contentComposite.setLayout(layout);
		return contentComposite;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarComboControl#createControls(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	protected void createControls(@NonNull Composite composite) {
		combo = createComboBox(composite);
		addButton = createAddButton(composite);
		if (SWToolsProperties.isVerificationOn()) {
			List<@NonNull String> componentsIds = getChild().getFeatureComponentAddValues();
			if (componentsIds != null) {
				CollectionsUtils.getInstancesOf(componentsIds.stream()
						.map(id -> controllerWrapper.getController().getConfigurationComponent(id)), ConfigurationComponent.class)
				.filter(c -> c.isGlobalOnly())
				.forEach(c -> LOGGER.warning("[DATA] Component type \"" + c.getId() + "\" referenced in enum with id \"" + child.getId() + "\" is not instantiable.")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.periphs.gui.view.componentsettings.ScalarComboControl#addSpecificListeners()
	 */
	@Override
	protected void addSpecificListeners() {
		Button addButtonLoc = addButton;
		if (addButtonLoc != null) {
			addButtonLoc.addSelectionListener(new SelectionAdapter() {
				/* (non-Javadoc)
				 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
				 */
				@Override
				public void widgetSelected(SelectionEvent e) {
					IViewPart viewPart = ComponentSettingViewHelper.getViewPart();
					if (viewPart == null) {
						return;
					}
					IViewSite viewSite = viewPart.getViewSite();
					if (viewSite == null) {
						return;
					}
					//FIXME TomasR v14 maintenance - replace this code with guiController call
					controllerWrapper.getController().runTransaction(() -> {
						List<@NonNull ConfigurationComponentTypeId> componentsThatPassedFilter =
								controllerWrapper.getController().getComponentTypeIdsForComponentAddFeature(getChild());
						AddComponentDialog openForComponents = AddComponentDialog.openForComponents(viewSite, componentsThatPassedFilter, controllerWrapper);
						IChildProvidable result = openForComponents.getResult();
						if (result == null) {
							return;
						}
						// newly created component/component instance should be selected in a combo
						ProgressUtils.run(m -> controllerWrapper.getController().setValue(getChild(), result.getUiName(), this));
					});
				}
			});
		}
	}

	/**
	 * Method used to create add button
	 * @param composite in which to create the add button
	 * @return add button
	 */
	private Button createAddButton(@NonNull Composite composite) {
		Button button = new Button(composite, SWT.PUSH);
		SWTFactoryProxy.INSTANCE.setTestId(button, TestIDs.PERIPHS_ADD_COMPONENT_INSTANCE_IN_COMPONENT_SETTING_VIEW_BUTTON + child.getId());
		SWTFactoryProxy.INSTANCE.setHtmlTooltip(button, Messages.get().ComponentsView_AddButtonTooltip);
		button.setImage(ToolsImages.getImage(IToolsImages.ICON_ADD_CIRCLE));
		button.setLayoutData(new GridDataComponents(SWT.RIGHT, SWT.FILL, false, false));
		button.setEnabled(!controllerWrapper.getController().getComponentTypeIdsForComponentAddFeature(getChild()).isEmpty());
		return button;
	}
}
