/**
* Copyright 2018-2022 NXP
* Created: Feb 6, 2018
*/
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Menu;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.logging.LogManager;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.gui.Messages;
import com.nxp.swtools.periphs.gui.view.componentsettings.ArrayControl;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl.UpdateType;
import com.nxp.swtools.provider.analytics.ActionAnalyticsBuilder;
import com.nxp.swtools.provider.configuration.ErrorLevels;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.ISettingConfig;
import com.nxp.swtools.resourcetables.model.validation.ValidationHelper;
import com.nxp.swtools.validation.engine.IBaseProblem;
import com.nxp.swtools.validation.engine.IValidationProblem;
import com.nxp.swtools.validation.utils.ValidationProblemsMenuHelper;

/**
 * A pop-up menu for items in {@link ArrayControl}
 * @author Viktar Paklonski
 */
public class ArrayControlItemMenu {
	/** Logger of the class */
	protected static final Logger LOGGER = LogManager.getLogger(ArrayControlItemMenu.class);
	/** instance owner */
	protected @NonNull AArrayControlInternal owner;
	/** menu control */
	protected @NonNull IArrayControlItemMenuControl control;
	/** Manager of the menu */
	private @NonNull MenuManager menuManager;
	/** Parent control for which the menu is created */
	private @NonNull Control parent;
	/** Current menu listener */
	private IMenuListener menuListener;

	/**
	 * Prepares menu for the specified widget.
	 * @param owner of the instance
	 * @param control of the menu
	 * @param parent a control which will be the parent of the new menu instance
	 */
	public ArrayControlItemMenu(@NonNull AArrayControlInternal owner, @NonNull IArrayControlItemMenuControl control, @NonNull Control parent) {
		this.owner = owner;
		this.control = control;
		this.parent = parent;
		MenuManager menuManagerLoc = new MenuManager("#Popup"); //$NON-NLS-1$
		menuManagerLoc.setRemoveAllWhenShown(true);
		menuManager = menuManagerLoc;
		fillManager(UpdateType.NORMAL);
	}

	/**
	 * Destroys private members.
	 */
	public void dispose() {
		menuManager.dispose();
	}

	/**
	 * @return {@link #menu}
	 */
	public @NonNull Menu getMenu() {
		return menuManager.createContextMenu(parent);
	}

	/**
	 * Sets menu location under the specified bounds and marks it as visible.
	 * @param bounds of the control
	 * @param parent of the control
	 * @param display of the widget
	 */
	public void showUnder(@NonNull Rectangle bounds, @NonNull Control parent, @NonNull Display display) {
		Menu menuLoc = getMenu();
        Event event = new Event();
        event.x = bounds.x;
        event.y = bounds.y + bounds.height;
        menuLoc.setVisible(true);
        menuLoc.notifyListeners(SWT.Show, event);
	}

	/**
	 * Fills the menu manager with listener which provides actions based on which setting is selected
	 * @param updateType updateType or {@code null} if update type is unknown
	 */
	public void fillManager(@Nullable UpdateType updateType) {
		// Clear before adding actions for current selection
		menuManager.removeAll();
		if (menuListener != null) {
			menuManager.removeMenuListener(menuListener);
		}
		IMenuListener listenerLoc = new IMenuListener() {
			@Override
			public void menuAboutToShow(IMenuManager manager) {
				final boolean allowed = (updateType != UpdateType.FORCE_DISABLE);
				final boolean first = owner.isControlFirst(control.getSelectedItemHint());
				final boolean last = owner.isControlLast(control.getSelectedItemHint());
				boolean isArrayHorizontal = owner.getControlOptions().isArrayHorizontal();
				String message;
				// Quick fixes addition
				IChild selectedSetting = control.getSelectedSettingHint();
				if (selectedSetting != null) {
					List<@NonNull IBaseProblem> problems = new ArrayList<>();
					problems.addAll(ValidationHelper.getSettingValidationProblems(selectedSetting, ErrorLevels.LEVEL_FATAL));
					problems.addAll(ValidationHelper.getSettingValidationProblems(selectedSetting, ErrorLevels.LEVEL_ERROR));
					problems.addAll(ValidationHelper.getSettingValidationProblems(selectedSetting, ErrorLevels.LEVEL_WARNING));
					List<@NonNull IBaseProblem> mergedProblems = new ArrayList<>(problems.size());
					ValidationProblemsMenuHelper.mergeProblems(problems, mergedProblems);
					if (!mergedProblems.isEmpty()) {
						for (IBaseProblem problem : mergedProblems) {
							if (problem instanceof IValidationProblem) {
								IValidationProblem validationProblem = (IValidationProblem) problem;
								MenuManager problemMenuManager = new MenuManager(validationProblem.getDependency().getDescription());
								ValidationProblemsMenuHelper.createPopupMenu(validationProblem, problemMenuManager, false);
								manager.add(problemMenuManager);
							}
						}
						manager.add(new Separator());
					}
				}
				// Move actions
				message = (isArrayHorizontal ? UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_MoveLeft) : UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_MoveUp));
				manager.add(createMenuManagerItem(() -> owner.moveItemFront(ArrayControlItemMenu.this, control), message, allowed && owner.isUiArrayReorderSpecified() && !first));
				message = (isArrayHorizontal ? UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_MoveRight) : UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_MoveDown));
				manager.add(createMenuManagerItem(() -> owner.moveItemBack(ArrayControlItemMenu.this, control), message, allowed && owner.isUiArrayReorderSpecified() && !last));
				message = (isArrayHorizontal ? UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_MoveLeftmost) : UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_MoveTop));
				manager.add(createMenuManagerItem(() -> owner.moveItemBeginning(ArrayControlItemMenu.this, control), message, allowed && owner.isUiArrayReorderSpecified() && !first));
				message = (isArrayHorizontal ? UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_MoveRightmost) : UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_MoveBottom));
				manager.add(createMenuManagerItem(() -> owner.moveItemEnd(ArrayControlItemMenu.this, control), message, allowed && owner.isUiArrayReorderSpecified() && !last));
				// Set all columns/row to this value
				if (control.getSelectedSettingHint() != null) {
					manager.add(new Separator());
					if (isArrayHorizontal) {
						message = UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_SetAllCols);
					} else {
						message = UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_SetAllRows);
					}
					if (owner.isSetAllSettingsToSupported(control)) {
						manager.add(createMenuManagerItem(() -> owner.setAllSettingsTo(ArrayControlItemMenu.this, control), message,
								(selectedSetting != null) && selectedSetting.isAvailable()));
					}
				}
				manager.add(new Separator());
				// Remove action
				manager.add(createMenuManagerItem(() -> owner.removeItem(ArrayControlItemMenu.this, control), UtilsText.safeString(Messages.get().ArrayControl_ItemMenu_Remove),
						allowed && !owner.isUiArrayFixedSpecified() && owner.getChild().canRemoveItem()));
				// Copy and paste actions
				ISettingConfig currentItemConfig = owner.getCurrentItemConfig(control);
				if (currentItemConfig != null) {
					String clipboardString = UtilsText.EMPTY_STRING;
					final Clipboard clipBoard = new Clipboard(getMenu().getDisplay());
					Object clipboardContent = clipBoard.getContents(TextTransfer.getInstance());
					if (clipboardContent instanceof String) {
						clipboardString = (String) clipboardContent;
					}
					final String clipboardStringFinal = clipboardString;
					Runnable method = () -> {
						String stringCopyResult = owner.getControllerWrapper().getController().copy(currentItemConfig);
						if (UtilsText.isEmpty(stringCopyResult)) {
							LOGGER.log(Level.SEVERE, "Copy did not produce String for the clipboard"); //$NON-NLS-1$
							return;
						}
						clipBoard.setContents(new Object[] { stringCopyResult }, new Transfer[] { TextTransfer.getInstance() });
						};
						manager.add(createMenuManagerItem(method, UtilsText.safeString(Messages.get().ChildControl_Copy), allowed && owner.getControllerWrapper().getController().canCopy(currentItemConfig)));
						manager.add(createMenuManagerItem(() -> owner.getControllerWrapper().getController().paste(currentItemConfig, clipboardStringFinal),
								UtilsText.safeString(Messages.get().ChildControl_Paste), allowed &&
								owner.getControllerWrapper().getController().canPaste(currentItemConfig, clipboardStringFinal)));
				}
			}
		};
		menuListener = listenerLoc;
		menuManager.addMenuListener(listenerLoc);
	}

	/**
	 * Creates action for context menu manager
	 * @param runnable to run when action is performed
	 * @param label of the action
	 * @param enabled {@code true} when action should be enabled in context menu, {@code false} when not
	 * @return the created action
	 */
	@NonNull IAction createMenuManagerItem(@NonNull Runnable runnable, @NonNull String label, boolean enabled) {
		return createMenuManagerItem(runnable, label, enabled, null, null, null);
	}

	/**
	 * Creates action for context menu manager
	 * @param runnable to run when action is performed
	 * @param label of the action
	 * @param enabled {@code true} when action should be enabled in context menu, {@code false} when not
	 * @param tooltip tooltip of the action. May be {@code null}
	 * @param testId test id of the action. May be {@code null}
	 * @param imageDescriptor descriptor of image for the action. May be {@code null}
	 * @return the created action
	 */
	@NonNull IAction createMenuManagerItem(@NonNull Runnable runnable, @NonNull String label, boolean enabled, @Nullable String tooltip, @Nullable String testId, @Nullable ImageDescriptor imageDescriptor) {
		ActionAnalyticsBuilder builder = ActionAnalyticsBuilder.action(runnable)
				.text(label)
				.enabled(enabled)
				.image(imageDescriptor);
		if (tooltip != null) {
			builder.tooltip(tooltip);
		}
		if (testId != null) {
			builder.id(testId);
		}
		return builder.build();
	}
}
