/**
 * Copyright 2017-2022 NXP
 * Created: 15. 12. 2017
 */
package com.nxp.swtools.periphs.gui.view.componentsettings.internal;

import java.util.ArrayList;
import java.util.List;

import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.stream.CollectorsUtils;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.gui.view.componentsettings.IChildControl;
import com.nxp.swtools.resourcetables.model.config.ArrayConfig;
import com.nxp.swtools.resourcetables.model.config.IChild;
import com.nxp.swtools.resourcetables.model.config.IRoot;
import com.nxp.swtools.resourcetables.model.config.ISettingConfig;
import com.nxp.swtools.resourcetables.model.data.SettingOptions;
import com.nxp.swtools.resourcetables.model.data.setting.IBaseModel;

/**
 * Class, which transforms horizontal IChildControl Lists into vertical ISettingConfig
 * @author Marek Ciz
 */
public class ArrayControlTabularHorizontalProvider {
	/** Identification for indices row */
	public static final @NonNull String INDICES_HEADER_IDENTIFIER = "#indices#"; //$NON-NLS-1$
	/** Identification for state row */
	public static final @NonNull String STATE_HEADER_IDENTIFIER = "#state#"; //$NON-NLS-1$
	/** Already vertically transformed children */
	@NonNull List<@NonNull List<@NonNull IChild>> transformedChildren;
	/** Original children */
	@NonNull List<@NonNull IChildControl> children;
	/** Extracted UI values from original children */
	@NonNull List<@NonNull List<@NonNull IChild>> flattenedChildren;
	/** Class representing configuration of an array setting */
	@NonNull ArrayConfig arrayConfig;
	/** Menu control UI label in the component table */
	static final @NonNull String MENU_HEADER_TITLE = UtilsText.EMPTY_STRING;
	/** Header names that are displayed in table */
	@NonNull List<@NonNull String> displayedHeaderNames;
	/** List of ignored rows */
	private @NonNull List<@NonNull Integer> ignoredRows;
	/** List of objects, where are stored SettingConfig lists. Flag if current setting should be used for remove button */
	@NonNull List<@NonNull TabularHorizontalItem> horizontalItems;

	/**
	 * Constructor.
	 * @param arrayConfig Class representing configuration of an array setting
	 * @param children Original children
	 */
	protected ArrayControlTabularHorizontalProvider(@NonNull ArrayConfig arrayConfig, @NonNull List<@NonNull IChildControl> children) {
		transformedChildren = new ArrayList<>();
		flattenedChildren = new ArrayList<>();
		this.arrayConfig = arrayConfig;
		this.children = children;
		// FIXME TomasR v13 maintenance - Refactor to use one public update function, because some lists are just temporary and not needed afterwards. The only public function to get output is getHorizontalItems()
		ignoredRows = getSkippedRows();
		displayedHeaderNames = getHeaders();
		transformedChildren = getFilledChildren();
		horizontalItems = fillHorizontalItems();
	}

	/**
	 * Creates transformed list structure
	 * @param rowCount total row count of a table
	 * @return empty list in required size
	 */
	private static @NonNull List<@NonNull List<@NonNull IChild>> createTransformedChildrenList(int rowCount) {
		@NonNull List<@NonNull List<@NonNull IChild>> transformedChildrenEmpty = new ArrayList<>();
		for (int i = 0; i < rowCount; i++) {
			transformedChildrenEmpty.add(new ArrayList<>());
		}
		return transformedChildrenEmpty;
	}

	/**
	 * Fill TabularHorizontalItems with proper values
	 * @return list of horizontal items
	 */
	private @NonNull List<@NonNull TabularHorizontalItem> fillHorizontalItems() {
		 @NonNull List<@NonNull TabularHorizontalItem> horizontalItemsLoc = new ArrayList<>();
		 @NonNull List<@NonNull String> headerNames = getHeaders();
		 int rowCount = headerNames.size() - 1;
		 horizontalItemsLoc.add(new TabularHorizontalItem(STATE_HEADER_IDENTIFIER, children.stream().map(x -> x.getChild()).collect(CollectorsUtils.toList())));
		 if (shouldShowIndices()) {
			 horizontalItemsLoc.add(new TabularHorizontalItem(INDICES_HEADER_IDENTIFIER, children.stream().map(x -> x.getChild()).collect(CollectorsUtils.toList())));
		 }
		 for (int row = 0; row < rowCount; row++) {
			 if (ignoredRows.contains(Integer.valueOf(row))) {
				 continue; // Row is ignored -> do not create item for it
			 }
			 // fill rows with proper data
			 horizontalItemsLoc.add(new TabularHorizontalItem(headerNames.get(row), transformedChildren.get(row)));
		 }
		 return horizontalItemsLoc;
	}

	/**
	 * @return check whether indices of the array should be shown
	 */
	protected boolean shouldShowIndices() {
		return !(arrayConfig.isOptionAvailable(SettingOptions.UI_ARRAY_INDICES_HIDDEN) && arrayConfig.isOptionSet(SettingOptions.UI_ARRAY_INDICES_HIDDEN));
	}

	/**
	 * Tabular Horizontal Items getter
	 * @return Tabular Horizontal Items
	 */
	public @NonNull List<@NonNull TabularHorizontalItem> getHorizontalItems() {
		return horizontalItems;
	}

	/**
	 * Extracts ISettingConfig from IChildControl and add it to the list
	 * @return flattened list of children
	 */
	private @NonNull List<@NonNull List<@NonNull IChild>> getFlattenedChildren() {
		@NonNull List<@NonNull List<@NonNull IChild>> flattenChildrenLoc = new ArrayList<>();
		for (IChildControl childControl : children) {
			flattenChildrenLoc.add(AArrayControlTabular.getSettingsFlat(childControl.getChild()));
		}
		return flattenChildrenLoc;
	}

	/**
	 * Extracts list of table headers
	 * @return list of table columns(rows - in horizontal layout) headers
	 */
	@NonNull List<@NonNull String> getHeaders() {
		// create some temporary fake settingConfig to retrieve proper column titles
		final IRoot root = arrayConfig.getChildContext().getRoot();
		@NonNull ISettingConfig childForHeaderLabels = root.getConfigFactory().createSettingConfig(UtilsText.EMPTY_STRING, UtilsText.EMPTY_STRING,
				arrayConfig.getModelData().getReferenceType(), arrayConfig.getChildContext(),
				root.getMcu());
		childForHeaderLabels.delayedInitAfterCreation();
		
		// original headers, that should be displayed in the first column
		@NonNull List<@NonNull String> originalHeaderNames = new ArrayList<>();
		for (IChild originalHeader : AArrayControlTabular.getSettingsFlat(childForHeaderLabels)) {
			IBaseModel baseModel = originalHeader.getModelData();
			if ((baseModel != null) && !baseModel.isOptionAvailable(SettingOptions.UI_NOT_VISIBLE_PERMANENT, arrayConfig.getExpressionContext())) {
				originalHeaderNames.add(baseModel.getUIName(arrayConfig.getExpressionContext()));
			}
		}
		// also add menu column title
		originalHeaderNames.add(MENU_HEADER_TITLE);
		// remove fake settings
		arrayConfig.remove(childForHeaderLabels);
		return originalHeaderNames;
	}

	/**
	 * Returns list of rows to be skipped
	 * @return list of rows to be skipped
	 */
	@NonNull List<@NonNull Integer> getSkippedRows() {
		// create some temporary fake settingConfig to retrieve proper column titles
		final IRoot root = arrayConfig.getChildContext().getRoot();
		@NonNull ISettingConfig childForHeaderLabels = root.getConfigFactory().createSettingConfig(UtilsText.EMPTY_STRING, UtilsText.EMPTY_STRING,
				arrayConfig.getModelData().getReferenceType(), arrayConfig.getChildContext(), root.getMcu());
		childForHeaderLabels.delayedInitAfterCreation();
		List<@NonNull IChild> settingsFlat = AArrayControlTabular.getSettingsFlat(childForHeaderLabels);
		@NonNull List<@NonNull Integer> skippedRows = new ArrayList<>();
		for (int i = 0; i < settingsFlat.size(); i++) {
			IChild originalHeader = settingsFlat.get(i);
			IBaseModel baseModel = originalHeader.getModelData();
			if ((baseModel != null) && baseModel.isOptionAvailable(SettingOptions.UI_NOT_VISIBLE_PERMANENT, arrayConfig.getExpressionContext())) {
				skippedRows.add(Integer.valueOf(i));
			}
		}
		// remove fake settings
		arrayConfig.remove(childForHeaderLabels);
		return skippedRows;
	}

	/**
	 * Fills vertically transformed children list with extracted Settings data
	 * @return transformed children list
	 */
	private @NonNull List<@NonNull List<@NonNull IChild>> getFilledChildren() {
		// rowCount stores how many rows are in a new table
		int rowCount = displayedHeaderNames.size() - 1;
		@NonNull List<@NonNull List<@NonNull IChild>> transformedChildrenLoc = createTransformedChildrenList(rowCount);
		flattenedChildren = getFlattenedChildren();
		// transform horizontal list to a vertical one
		if (!flattenedChildren.isEmpty()) {
			for (int row = 0; row < rowCount; row++) {
				for (int col = 0; col < children.size(); col++) {
					transformedChildrenLoc.get(row).add(flattenedChildren.get(col).get(row));
				}
			}
		}
		return transformedChildrenLoc;
	}

	/**
	 * @return list of transformed children
	 */
	protected @NonNull List<@NonNull List<@NonNull IChild>> getChildren() {
		return transformedChildren;
	}
}
