/**
 * Copyright 2017-2022 NXP
 */
package com.nxp.swtools.periphs.gui.view.provider;

import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import com.nxp.swtools.codemanager.GeneratedCodeStatus;
import com.nxp.swtools.common.utils.NonNull;
import com.nxp.swtools.common.utils.Nullable;
import com.nxp.swtools.common.utils.text.UtilsText;
import com.nxp.swtools.periphs.controller.Controller;
import com.nxp.swtools.periphs.controller.Messages;
import com.nxp.swtools.periphs.model.data.mcu.IMcu;
import com.nxp.swtools.provider.configuration.SharedConfigurationFactory;
import com.nxp.swtools.provider.configuration.sources.ISourceFile;
import com.nxp.swtools.provider.configuration.sources.SourceFileProviderFactory;
import com.nxp.swtools.resourcetables.model.mcu.ACommonMcu;
import com.nxp.swtools.utils.scripting.ScriptResult;
import com.nxp.swtools.utils.sources.ISourcesContentProvider;

/**
 * Class used for providing content to the Sources View.
 * @author Juraj Ondruska
 */
public class SourcesViewProvider implements ISourcesContentProvider {

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.sources.ISourcesContentProvider#getContent()
	 */
	@Override
	public @NonNull Collection<@NonNull ISourceFile> getContent() {
		Controller controller = Controller.getInstance();
		ScriptResult scriptResult = controller.getGeneratedCode();
		Map<@NonNull String, @NonNull String> code = new LinkedHashMap<>();
		Map<@NonNull String, byte @NonNull[]> binaryFiles = new LinkedHashMap<>();
		if (scriptResult != null) {
			code.putAll(scriptResult.getOpenedFiles()); // add all core-unrelated files
			binaryFiles.putAll(scriptResult.getOpenedBinaryFiles()); // add all core-unrelated files
			IMcu mcu = controller.getMcu();
			for (String core : scriptResult.getCores()) {
				Map<@NonNull String, @NonNull String> coreFiles = scriptResult.containsAnyError() ? new HashMap<>() : scriptResult.getOpenedFiles(core);
				Map<@NonNull String, byte @NonNull[]> coreBinaryFiles = scriptResult.containsAnyError() ? new HashMap<>() : scriptResult.getOpenedBinaryFiles(core);
				if (coreFiles != null) {
					for (Entry<@NonNull String, @NonNull String> entry : coreFiles.entrySet()) {
						code.put(ACommonMcu.addCoreNameIfMulticore(mcu, core, entry.getKey()), entry.getValue());
					}
				}
				if (coreBinaryFiles != null) {
					for (Entry<@NonNull String, byte @NonNull[]> entry : coreBinaryFiles.entrySet()) {
						binaryFiles.put(ACommonMcu.addCoreNameIfMulticore(mcu, core, entry.getKey()), entry.getValue());
					}
				}
			}
		}
		return SourceFileProviderFactory.buildSourceFileList(code, binaryFiles,
				SharedConfigurationFactory.getSharedConfigurationSingleton().getPreferences().getCustomCopyright());
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.sources.ISourcesContentProvider#getErrorText()
	 */
	@Override
	public @Nullable String getErrorText() {
		final Controller controller = Controller.getInstance();
		if (!controller.isTurnedOn() || controller.isGeneratingCode()) {
			return null;
		}
		ScriptResult scriptResult = controller.getGeneratedCode();
		String mcuError = controller.getToolError();
		if (mcuError != null) {
			return mcuError;
		}
		if ((scriptResult != null) && scriptResult.isFailedDueTimeout()) {
			return com.nxp.swtools.periphs.controller.Messages.get().Controller_SourcesGeneration_GenerationFailedDueTimeout;
		}
		return ((scriptResult != null) && !scriptResult.isFailed())
				? scriptResult.getErrors()
				: com.nxp.swtools.periphs.controller.Messages.get().Controller_SourcesGeneration_CannotGenerateDueToInternalError;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.sources.ISourcesContentProvider#getWarningText()
	 */
	@Override
	public @Nullable String getWarningText() {
		final Controller controller = Controller.getInstance();
		if (!controller.isTurnedOn() || controller.isGeneratingCode()) {
			return null;
		}
		ScriptResult scriptResult = controller.getGeneratedCode();
		return scriptResult != null ? scriptResult.getWarnings() : null;
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.sources.ISourcesContentProvider#isOutOfSync()
	 */
	@Override
	public boolean isOutOfSync() {
		Controller controller = Controller.getInstance();
		ScriptResult generatedCode = controller.getGeneratedCode();
		return ((controller.isGeneratingCode()) || ((generatedCode != null) ? generatedCode.isFailed() : true)
				|| GeneratedCodeStatus.getInstance().isOutOfSync());
	}

	/* (non-Javadoc)
	 * @see com.nxp.swtools.utils.sources.ISourcesContentProvider#getStatus()
	 */
	@Override
	public @Nullable String getStatus() {
		final Controller controller = Controller.getInstance();
		if (!controller.isTurnedOn() || controller.isGeneratingCode()) {
			return null;
		}
		ScriptResult scriptResult = controller.getGeneratedCode();
		String mcuError = controller.getToolError();
		if (mcuError != null) {
			return mcuError;
		} else if ((scriptResult != null) && scriptResult.isFailedDueTimeout()) {
			return Messages.get().Controller_SourcesGeneration_GenerationFailedDueTimeout;
		} else if ((scriptResult == null) || scriptResult.isFailed() || scriptResult.containsAnyError()) {
			return Messages.get().Controller_SourcesGeneration_CannotGenerateDueToInternalError;
		} else if (!UtilsText.isEmpty(scriptResult.getErrors())) {
			return Messages.get().SourcesViewProvider_SourcesGeneration_GeneratedWithErrors;
		} else if (!UtilsText.isEmpty(scriptResult.getWarnings())) {
			return Messages.get().SourcesViewProvider_SourcesGeneration_GeneratedWithWarnings;
		}
		return UtilsText.EMPTY_STRING;
	}

}
