/*******************************************************************************
 * Copyright (c) 2001, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package com.freescale.system.browser.epl;

import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.forms.IFormColors;

import com.freescale.system.browser.epl.ITabbedConstantsEpl;
import com.freescale.system.browser.epl.TabbedWidgetFactoryEpl;


/**
 * The title in the tabbed sheet page.
 */
public class TabbedTitleEpl extends Composite {

	/**
	 * Blank string used as default title text.
	 */
	private static final String BLANK = ""; //$NON-NLS-1$
	
	/**
	 * The label used for the title.
	 */
	private CLabel fLabel;

	/**
	 * The image from the title.
	 */
	private Image fImage = null;

	/**
	 * The text from the title
	 */
	private String fText = null;

	/**
	 * Width of the margin that will be added around the control.
	 */
	private int fMarginWidth = 4;

	/**
	 * Height of the margin that will be added around the control.
	 */
	private int fMarginHeight = 4;

	/**
	 * The widget factory used to instantiate the widgets.
	 */
	private TabbedWidgetFactoryEpl fFactory;

	/**
	 * Constructor for TabbedTitle.
	 * 
	 * @param parent
	 *            the parent composite.
	 * @param factory
	 *            the widget factory for the tabbed property sheet
	 */
	public TabbedTitleEpl(Composite parent,
			TabbedWidgetFactoryEpl factory) {
		super(parent, SWT.NO_FOCUS);
		fFactory = factory;

		this.addPaintListener(new PaintListener() {
			@Override  
			public void paintControl(PaintEvent e) {
				if (fImage == null && (fText == null || fText.equals(BLANK))) {
					fLabel.setVisible(false);
				} else {
					fLabel.setVisible(true);
					drawTitleBackground(e);
				}
			}
		});

		factory.getColors().initializeSectionToolBarColors();
		setBackground(factory.getColors().getBackground());
		setForeground(factory.getColors().getForeground());

		FormLayout layout = new FormLayout();

		layout.marginWidth = ITabbedConstantsEpl.HSPACE + 6;
		layout.marginHeight = 5;
		setLayout(layout);

		fLabel = factory.createCLabel(this, BLANK);
		fLabel.setBackground(new Color[] {
			factory.getColors().getColor(IFormColors.TB_BG), factory.getColors().getColor(IFormColors.TB_BG)},
				new int[] { 100}, true);
		fLabel.setFont(JFaceResources.getBannerFont());
		FormData data = new FormData();

		data.left = new FormAttachment(0, 0);
		data.top = new FormAttachment(0, 0);
		data.right = new FormAttachment(100, 0);
		data.bottom = new FormAttachment(100, 0);
		fLabel.setLayoutData(data);
	}

	/**
	 * Set the text and image 
	 */
	public void setTitle(String text, Image image) {
		this.fText = text;
		this.fImage = image;
		if (text != null) {
			fLabel.setText(text);
		} else {
			fLabel.setText(BLANK); 
		}
		fLabel.setImage(image);
		redraw();
	}

	/**
	 * Return the height of the title.
	 */
	public int getHeight() {
		Shell shell = new Shell();
		GC gc = new GC(shell);

		gc.setFont(getFont());
		Point point = gc.textExtent(BLANK);

		point.x++;
		int textOrImageHeight = Math.max(point.x, 16);

		gc.dispose();
		shell.dispose();
		return textOrImageHeight + 8;
	}
	
	/**
	 * Draws the title background.
	 * 
	 * @param e the paint event
	 */
	protected void drawTitleBackground(PaintEvent e) {
		Color bg = fFactory.getColors().getColor(IFormColors.TB_BG);
		Color gbg = fFactory.getColors().getColor(IFormColors.TB_BG);
		Color border = fFactory.getColors().getColor(IFormColors.TB_BORDER);
		Rectangle bounds = getClientArea();
		//Point tsize = null;
		Point labelSize = null;
		//int twidth = bounds.width - fMarginWidth - fMarginWidth;

		if (fLabel != null) {
			labelSize = fLabel.computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
		}
		
		// local variable twidth not used!
		//if (labelSize != null) {
		//twidth -= labelSize.x + 4;
		//}
		int tvmargin = 4;
		int theight = getHeight();

		// following code has no effect as tsize is always null!
		/*if (tsize != null) {
		 theight += Math.max(theight, tsize.y);
		 }*/
		
		if (labelSize != null) {
			theight = Math.max(theight, labelSize.y);
		}
		theight += tvmargin + tvmargin;
		int midpoint = (theight * 66) / 100;
		int rem = theight - midpoint;
		GC gc = e.gc;

		gc.setForeground(bg);
		gc.setBackground(gbg);
		gc.fillGradientRectangle(fMarginWidth, fMarginHeight, bounds.width - 1 - fMarginWidth - fMarginWidth,
				midpoint - 1, true);
		gc.setForeground(gbg);
		gc.setBackground(getBackground());
		gc.fillGradientRectangle(fMarginWidth, fMarginHeight + midpoint - 1,
				bounds.width - 1 - fMarginWidth - fMarginWidth, rem - 1, true);
		gc.setForeground(border);
		gc.drawLine(fMarginWidth, fMarginHeight + 2, fMarginWidth, fMarginHeight + theight - 1);
		gc.drawLine(fMarginWidth, fMarginHeight + 2, fMarginWidth + 2, fMarginHeight);
		gc.drawLine(fMarginWidth + 2, fMarginHeight, bounds.width - fMarginWidth - 3, fMarginHeight);
		gc.drawLine(bounds.width - fMarginWidth - 3, fMarginHeight, bounds.width - fMarginWidth - 1, fMarginHeight + 2);
		gc.drawLine(bounds.width - fMarginWidth - 1, fMarginHeight + 2, bounds.width - fMarginWidth - 1,
				fMarginHeight + theight - 1);
	}
}
