/*******************************************************************************
 Copyright 2023 NXP 

 NXP Confidential and Proprietary. This software is owned or controlled by NXP
 and may only be used strictly in accordance with the applicable license terms.
 By expressly accepting such terms or by downloading, installing, activating
 and/or otherwise using the software, you are agreeing that you have read, and
 that you agree to comply with and are bound by, such license terms. If you do
 not agree to be bound by the applicable license terms, then you may not retain,
 install, activate or otherwise use the software.
 *******************************************************************************/

package com.freescale.system.browser.epl.help;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.part.Page;

import com.freescale.system.browser.epl.AbstractSectionEpl;
import com.freescale.system.browser.epl.help.TasksTabDescriptorEpl.Column;
import com.freescale.system.browser2.ISimpleTableDataProvider2;
import com.freescale.system.browser2.Messages2;
import com.freescale.system.browser2.OSAwarenessData2;
import com.freescale.system.browser2.SystemBrowserActivator2;
import com.freescale.system.browser2.TasksTreeContentProvider2;

public class TasksSectionEpl extends AbstractSectionEpl {
	private static final int DEFAULT_COLUMN_WIDTH = 100;

	/**
	 * The jFace table; the only control in this section 
	 */
	private TreeViewer fTasksTableViewer;

	/**
	 * Custom columns the System Contributor wants us to show; requested
	 * via the extension. Will be an empty array if non are requested. 
	 */
	private Column[] fCustomColumns;

	/**
	 * The value passed into the last setInput method. May be different than
	 * the input we set into the viewer if we deem the difference between the
	 * two irrelevant. 
	 */
	private Object fInput;

	/**
	 * The toolbar button with the little green bug; let's the user target
	 * a process or thread that's not already targeted.
	 */
	//private DebugAction fDebugAction = new DebugAction();

	/**
	 * The system contributor can ask that we not show the name column via the
	 * extension XML. No point forcing it if the OS doesn't associate names.
	 */
	private boolean fHideNameColumn;

	/**
	 * The system contributor can specify the name-column width via the
	 * extension XML. Some OS's might restrict names to say 8 or 10 characters.
	 * Maybe others tend to have long names. Give the KA vendor control.
	 */
	private Integer fNameColumnWidth;

	/**
	 * The system contributor can specify that we not show the status column,
	 * via the extension XML. The reason a vendor would is that the column
	 * supports only two state: running or suspended. Many OS's have a richer
	 * range of states to report and can/should do so via a custom column. If
	 * they do, then showing the built-in column is mostly redundant if not
	 * confusing.
	 */
	private boolean fHideStatusColumn;

	/**
	 * The system contributor can specify that we not show the ID column,
	 * via the extension XML. The reason a vendor would is that it uses
	 * some fabricated identification system and exposes that to the 
	 * debugger as the PID (e.g., the address os some kernel structure)
	 * mostly because the PID alone doesn't uniquely identify the process
	 * in that environment (e.g., consider ASMP Linux). In this case, the 
	 * system contributor would likely want to hide the ID field and contribute 
	 * a custom column to show the PID. 
	 */
	private boolean fHideIdColumn;

	/**
	 * The system contributor can specify that the ID column show hex values
	 * instead of decimal ones.
	 */
	private boolean fHexIdColumn;	
	/**
	 * <p>A mouse listener that handles the right click and the double click.</p>
	 *
	 */
	private MouseAdapter fMouseListener = new MouseAdapter() {
		private static final int LEFT_BUTTON_CODE = 1;
		private static final int RIGHT_BUTTON_CODE = 3;
		
		public void mouseDoubleClick(MouseEvent e) {
			if (LEFT_BUTTON_CODE == e.button) {
				//fDebugAction.run();
			}
		}
		
		public void mouseDown(MouseEvent e) {
			if (RIGHT_BUTTON_CODE == e.button) {
				Object source = e.getSource();
				if (source instanceof Tree) {
					Tree tree = (Tree)source;
					createContextMenu(tree);
				}
			}
		}
		
		/**
		 * <p>Create a pop-up context menu.</p>
		 * 
		 * @param tree the tree widget under the mouse click
		 */
		private void createContextMenu(Tree tree) {
			MenuManager menuMgr = new MenuManager("#PopupMenu"); //$NON-NLS-1$
			menuMgr.setRemoveAllWhenShown(true);
			menuMgr.addMenuListener(new IMenuListener() {
				public void menuAboutToShow(IMenuManager m) {
					//m.add(fDebugAction);
				}
			});
			Menu menu = menuMgr.createContextMenu(tree);
			tree.setMenu(menu);
		}
	};
    /**  */
	private ISimpleTableDataProvider2 fTaskData;
    /**  */
	private TasksTreeContentProvider2 contentProvider;
	
	static public class Options {
		public boolean hideNameColumn;
		public Integer nameColumnWidth;
		public boolean hideStatusColumn;
		public boolean hideIdColumn;
		public boolean hexIdColumn;
		public ISimpleTableDataProvider2 taskData;
	}

	/**
	 * Constructor.
	 * 
	 * @param customColumns
	 *            descriptors of custom columns contributed by the system
	 *            browser via the extension XML
	 * @param options
	 *            options that can be optionally specified in the exension XML.
	 *            This param can be null.
	 * 
	 */
	public TasksSectionEpl(Column[] customColumns, Options options) {
		super();
		fCustomColumns = customColumns.clone();
		fInput = null;
		if (options != null) {
			fHideNameColumn = options.hideNameColumn;
			fNameColumnWidth = options.nameColumnWidth;
			fHideStatusColumn = options.hideStatusColumn;
			fHideIdColumn = options.hideIdColumn;
			fHexIdColumn = options.hexIdColumn;			
			fTaskData = options.taskData;
		}
	}

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.AbstractSection#shouldUseExtraSpace()
	 */
	public boolean shouldUseExtraSpace() {
		return true;
	}

	/**
	 * Creates the GUI controls of this section.
	 */
	@Override
	public void createControls(Composite parent, Page page) {
		//SystemBrowserActivator.log(1, "createControls =>"+page.toString()); //$NON-NLS-1$ 
		super.createControls(parent, page);

		Composite composite = getWidgetFactory().createComposite(parent);
		composite.setLayout(new GridLayout());

		fTasksTableViewer = new TreeViewer(composite, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
		fTasksTableViewer.getControl().setLayoutData(new GridData(GridData.FILL, GridData.FILL, true, true));
		
		Tree tree = fTasksTableViewer.getTree();
		tree.setHeaderVisible(true);	
		// Build up the complete list of columns: built-in and custom 
		List<Column> cols = new ArrayList<>();

		// The name column (can optionally be hidden). When content is displayed
		// as a tree (the case when fScopeIsMachine), the column needs to 
		// be wider since the sub-elements are indented (by about 30 pixels)
		if (!fHideNameColumn) {
			int width = DEFAULT_COLUMN_WIDTH;
			if (fNameColumnWidth != null) {
				width = fNameColumnWidth.intValue();
			}
			cols.add(new Column(Messages2.TasksTab_Name_Column, TasksTabDescriptorEpl.BUILTIN_COLUMN_DATAID_NAME, width));
		}

		// The ID column
		if (!fHideIdColumn) {
			// ID doesn't take much real estate, but more so if showing
			// a three and we're the first column (see comment for name column)
			int width = DEFAULT_COLUMN_WIDTH / 2; 
			cols.add(new Column(Messages2.TasksTab_ID_Column, TasksTabDescriptorEpl.BUILTIN_COLUMN_DATAID_ID, width)); 
		}

		// The status column
		if (!fHideStatusColumn) {
			cols.add(new Column(Messages2.TasksTab_Status_Column, TasksTabDescriptorEpl.BUILTIN_COLUMN_DATAID_STATUS, DEFAULT_COLUMN_WIDTH));			
		}

		// Custom columns
		for (Column customCol : fCustomColumns) {
			cols.add(customCol);
		}
		
		// Add the columns to the table control. We support sorting so add a selection listener
		int colIndex = 0;
		for (Column col : cols) {
			TreeColumn column = new TreeColumn(tree, SWT.LEFT);
			column.setText(col.label);
			column.setWidth(col.width);
			final int index = colIndex;
			column.addSelectionListener(new SelectionListener() {
				public void widgetDefaultSelected(SelectionEvent e) {}

				public void widgetSelected(SelectionEvent e) {
					TasksTreeComparatorEpl comparator = (TasksTreeComparatorEpl)fTasksTableViewer.getComparator();
					if (comparator != null) {
						comparator.setSortColumn(index);
						fTasksTableViewer.refresh(false);
					}
				}
			});
			colIndex++;
		}
        
		
		fTasksTableViewer.setLabelProvider(
				new TasksTreeLabelProviderEpl(cols.toArray(new Column[cols.size()]), fHexIdColumn));
		
		// Give the viewer a content and label provider
		contentProvider = new TasksTreeContentProvider2(fTasksTableViewer);
	    fTasksTableViewer.setContentProvider(contentProvider);		
		fTasksTableViewer.getTree().addMouseListener(fMouseListener);
		
		// set a comparator in the viewer; it will make the column sorting possible
		fTasksTableViewer.setComparator(new TasksTreeComparatorEpl(cols.toArray(new Column[cols.size()])));
	}	
	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.AbstractSection#aboutToBeShown()
	 */
	@Override
	public void aboutToBeShown() {
		super.aboutToBeShown();
		TasksTreeContentProvider2 contentProvider = (TasksTreeContentProvider2)fTasksTableViewer.getContentProvider();		
		contentProvider.viewerAboutToBeShown();		
	}
	
	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.AbstractSection#aboutToBeHidden()
	 */
	@Override
	public void aboutToBeHidden() {
		super.aboutToBeHidden();
		
		// If we're not showing, defer any refreshing of the viewer by the content provider.
		TasksTreeContentProvider2 contentProvider = (TasksTreeContentProvider2)fTasksTableViewer.getContentProvider();
		if (contentProvider != null) {
			contentProvider.viewerAboutToBeHidden();
		}
	}

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.AbstractSection#dispose()
	 */
	@Override
	public void dispose() {//
		// don't think we have anything to do here
		contentProvider.dispose();
	}

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.ISection#setInput(java.lang.Object)
	 */
	@Override
	public void setSectionInput(final OSAwarenessData2 input) {
		SystemBrowserActivator2.log(1, "setSectionInput =>"+input.toString()); //$NON-NLS-1$
		fInput = input;
		if (fTasksTableViewer == null) {
			return; // no GUI yet
		}
		if (input.getContext() != null) {			
			
			OSAwarenessData2 old_input = (OSAwarenessData2)fTasksTableViewer.getInput();
			if (old_input != null && old_input.equals(input)) {
				return;
			}
			contentProvider.registerSession(input.getContext().getSessionId());
		    input.setLink(fTaskData);
		    fTasksTableViewer.setInput(input);			
		} else {
			contentProvider.registerSession(null);
			input.setLink(null);
			 fTasksTableViewer.setInput(null);
		}		
	}
	
	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.ISection#getInput()
	 */
	@Override
	public Object getSectionInput() {
		return fInput;
	}

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.ISection#refresh()
	 */
	@Override
	public void refresh() {
		fTasksTableViewer.refresh();
		super.refresh();
	}

	/* (non-Javadoc)
	 * @see com.freescale.system.browser.ui.tabbed.ISection#setActionBarsForTab(com.freescale.system.browser.ui.tabbed.ITabbedPageContributor, org.eclipse.ui.IActionBars)
	 */
	@Override
	public void setActionBarsForTab(IActionBars actionBars) {
		//actionBars.getToolBarManager().add(fDebugAction);
	}
}
