/*******************************************************************************
 * Copyright (c) 2012, 2013 EclipseSource and others.
 * Copyright 2018, 2023 NXP
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    EclipseSource - initial API and implementation
 *    Freescale - copied without modification
 *    NXP - 2018 - added null check on attribute in function {@link #checkIntAttribute( String, Attributes, String, String)}
 ******************************************************************************/
package com.nxp.swtools.derivative.swt;

import java.io.StringReader;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.XMLConstants;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.eclipse.swt.widgets.Widget;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Class to validate text to rap markup in {@linkplain HtmlStyledText#convertToRapHtmlFormat(String)}
 * @author Jan Vanacek
 * @author Lukas Tyc - B45588
 * Taken from internal class MarkupValidator in org.eclipse.rap.rwt library, class is is not accessible outside library.
 */
public class RapMarkupValidator {
	private static RapMarkupValidator instance = null;
  // Used by Eclipse Scout project
  public static final String MARKUP_VALIDATION_DISABLED
    = "org.eclipse.rap.rwt.markupValidationDisabled"; //$NON-NLS-1$
  /** exception text of unsupported element */
  public static final String UNSUPPORTED_ELEMENT_IN_MARKUP_TEXT = "Unsupported element in markup text: "; //$NON-NLS-1$

  private static final String DTD = createDTD();
  private static final Map<String, String[]> SUPPORTED_ELEMENTS = createSupportedElementsMap();
  private final SAXParser saxParser;

	/**
	 * Constructor 
	 * <p> in most cases can use singleton {@link #getInstance()} </p>
	 */
	public RapMarkupValidator() {
    saxParser = createSAXParser();
  }
  
	/**
	 * <p>
	 * SAXException: FWK005 parse may not be called while parsing, fix by using {@link #RapMarkupValidator()}.
	 * </p>
	 * @return singleton instance
	 */
	public static RapMarkupValidator getInstance() {
      if(instance == null) {
         instance = new RapMarkupValidator();
      }
      return instance;
   }

	/**
	 * @param text validated text
	 * @throws IllegalArgumentException reason of the failed validation
	 */
	public void validate(String text) throws IllegalArgumentException {
		StringBuilder markup = new StringBuilder();
		markup.append(DTD);
		markup.append("<html>"); //$NON-NLS-1$
		markup.append(text);
		markup.append("</html>"); //$NON-NLS-1$
		InputSource inputSource = new InputSource(new StringReader(markup.toString()));
		try {
			saxParser.parse(inputSource, new MarkupHandler());
		} catch (RuntimeException exception) {
			throw exception;
		} catch (Exception exception) {
			throw new IllegalArgumentException("Failed to parse markup text", exception); //$NON-NLS-1$
		}
	}

	/**
	 * @param widget checked widget on markup validation
	 * @return <code>true</code> validation is disabled
	 */
	public static boolean isValidationDisabledFor(Widget widget) {
		return Boolean.TRUE.equals(widget.getData(MARKUP_VALIDATION_DISABLED));
	}

	private static SAXParser createSAXParser() {
		SAXParser result = null;
		SAXParserFactory parserFactory = SAXParserFactory.newInstance();
		try {
			parserFactory.setFeature("http://apache.org/xml/features/disallow-doctype-decl", true); //$NON-NLS-1$
			parserFactory.setFeature(XMLConstants.FEATURE_SECURE_PROCESSING, true);
			result = parserFactory.newSAXParser();
		} catch (Exception exception) {
			throw new RuntimeException("Failed to create SAX parser", exception); //$NON-NLS-1$
		}
		return result;
	}

  private static String createDTD() {
    StringBuilder result = new StringBuilder();
    result.append( "<!DOCTYPE html [" ); //$NON-NLS-1$
    result.append( "<!ENTITY quot \"&#34;\">" ); //$NON-NLS-1$
    result.append( "<!ENTITY amp \"&#38;\">" ); //$NON-NLS-1$
    result.append( "<!ENTITY apos \"&#39;\">" ); //$NON-NLS-1$
    result.append( "<!ENTITY lt \"&#60;\">" ); //$NON-NLS-1$
    result.append( "<!ENTITY gt \"&#62;\">" ); //$NON-NLS-1$
    result.append( "<!ENTITY nbsp \"&#160;\">" ); //$NON-NLS-1$
    result.append( "<!ENTITY ensp \"&#8194;\">" ); //$NON-NLS-1$
    result.append( "<!ENTITY emsp \"&#8195;\">" ); //$NON-NLS-1$
    result.append( "<!ENTITY ndash \"&#8211;\">" ); //$NON-NLS-1$
    result.append( "<!ENTITY mdash \"&#8212;\">" ); //$NON-NLS-1$
    result.append( "]>" ); //$NON-NLS-1$
    return result.toString();
  }

  private static Map<String, String[]> createSupportedElementsMap() {
    Map<String, String[]> result = new HashMap<String, String[]>();
    result.put( "html", new String[ 0 ] ); //$NON-NLS-1$
    result.put( "br", new String[ 0 ] ); //$NON-NLS-1$
    result.put( "b", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "strong", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "i", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "em", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "sub", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "sup", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "big", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "small", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "del", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "ins", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "code", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "samp", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "kbd", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "var", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "cite", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "dfn", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "q", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "abbr", new String[] { "style", "title" } ); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
    result.put( "span", new String[] { "style" } ); //$NON-NLS-1$ //$NON-NLS-2$
    result.put( "img", new String[] { "style", "src", "width", "height", "title", "alt" } ); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$ //$NON-NLS-7$
    result.put( "a", new String[] { "style", "href", "target", "title" } ); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
    return result;
  }

  private static class MarkupHandler extends DefaultHandler {

	@Override
    public void startElement( String uri, String localName, String name, Attributes attributes ) {
      checkSupportedElements( name, attributes );
      checkSupportedAttributes( name, attributes );
      checkMandatoryAttributes( name, attributes );
    }

    private static void checkSupportedElements( String elementName, Attributes attributes ) {
      if( !SUPPORTED_ELEMENTS.containsKey( elementName ) ) {
        throw new IllegalArgumentException( UNSUPPORTED_ELEMENT_IN_MARKUP_TEXT + elementName );
      }
    }

    private static void checkSupportedAttributes( String elementName, Attributes attributes ) {
      if( attributes.getLength() > 0 ) {
        List<String> supportedAttributes = Arrays.asList( SUPPORTED_ELEMENTS.get( elementName ) );
        int index = 0;
        String attributeName = attributes.getQName( index );
        while( attributeName != null ) {
          if( !supportedAttributes.contains( attributeName ) ) {
            String message = "Unsupported attribute \"{0}\" for element \"{1}\" in markup text"; //$NON-NLS-1$
            message = MessageFormat.format( message, new Object[] { attributeName, elementName } );
            throw new IllegalArgumentException( message );
          }
          index++;
          attributeName = attributes.getQName( index );
        }
      }
    }

    private static void checkMandatoryAttributes( String elementName, Attributes attributes ) {
      checkIntAttribute( elementName, attributes, "img", "width" ); //$NON-NLS-1$ //$NON-NLS-2$
      checkIntAttribute( elementName, attributes, "img", "height" ); //$NON-NLS-1$ //$NON-NLS-2$
    }

    private static void checkIntAttribute( String elementName,
                                           Attributes attributes,
                                           String checkedElementName,
                                           String checkedAttributeName )
    {
      if( checkedElementName.equals( elementName ) ) {
        String attribute = attributes.getValue( checkedAttributeName );
        Object[] arguments = new Object[] { checkedAttributeName, checkedElementName };
        if (attribute!=null) {
          try {
            Integer.parseInt( attribute );
          } catch( NumberFormatException exception ) {
            String message
              = "Mandatory attribute \"{0}\" for element \"{1}\" is not a valid integer"; //$NON-NLS-1$
            message = MessageFormat.format( message, arguments );
            throw new IllegalArgumentException( message, exception );
          }
        } else {
          String message
            = "Mandatory attribute \"{0}\" for element \"{1}\" is missing"; //$NON-NLS-1$
          message = MessageFormat.format( message, arguments );
          throw new IllegalArgumentException( message );
        }
      }
    }

  }

}
