/*
 * Copyright 2014-2015 Freescale
 * Copyright 2016-2018,2021,2023 NXP
 * SPDX-License-Identifier: EPL-1.0
 */

package com.nxp.sa.ui.common.epl;

import java.util.concurrent.CountDownLatch;

import com.freescale.sa.util.Logger;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IPathEditorInput;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.widgets.Form;
import org.eclipse.ui.help.IWorkbenchHelpSystem;
import org.eclipse.ui.ide.FileStoreEditorInput;
import org.eclipse.ui.part.EditorPart;
import org.eclipse.ui.part.FileEditorInput;

import com.freescale.sa.GCUtil;
import com.freescale.sa.util.CommonConstants;

/**
 * Groups common functionality of SA editors.
 * 
 * @author Radu Ivan
 * @version 1.0
 */

public abstract class AbstractSaEditor extends EditorPart implements IResourceChangeListener {
	
	protected static final String TRACE_FOLDER = "TraceData"; //$NON-NLS-1$
	public final static String trace_csv_ext = ".csv"; //$NON-NLS-1$
	public final static String trace_data_ext = ".dat"; //$NON-NLS-1$
	public final static String trace_tarmac_ext = ".tarmac"; //$NON-NLS-1$
	public final static String trace_cwzsa_ext = ".cwzsa"; //$NON-NLS-1$
	public final static String trace_kcwzsa_ext = ".kcwzsa"; //$NON-NLS-1$
	public final static String trace_scwzsa_ext = ".scwzsa"; //$NON-NLS-1$
	public final static String AbstractSaEditor_DefaultText = "Analysis Results"; //$NON-NLS-1$

	private static class RefreshListener implements IPartListener {

		/**
		 * Equivalent to focus gained - checks if an editor needs refresh.
		 */
		@Override
		public void partActivated(IWorkbenchPart part) {
			refreshEditors(part);
		}

		@Override
		public void partBroughtToTop(IWorkbenchPart part) {
		}

		@Override
		public void partClosed(IWorkbenchPart part) {
		}

		@Override
		public void partDeactivated(IWorkbenchPart part) {
		}

		@Override
		public void partOpened(IWorkbenchPart part) {
		}
		
		/***
		 * Check if the editor is derived from AbstractSaEditor and calls the refresh function.
		 * 
		 * @param partEd - the part which may need refresh.
		 */
		public void refreshEditors(IWorkbenchPart partEd) {
			try {
				if (partEd instanceof AbstractSaEditor) {
					// first get search for the editor
					AbstractSaEditor td = (AbstractSaEditor)partEd;
					if (td.needsRefresh()) {
						td.refresh();
						// clear refresh flag
						td.shouldRefresh(false);
					}
				}
			} catch (Exception ex) {
				ex.printStackTrace();
			}
		}
	}

	final private String m_id;
	final private String m_helpContextId;
	private boolean m_needsRefresh;
	protected IProject m_project = null;


	private CountDownLatch m_createSwtControlsReady = new CountDownLatch(1);

	/**
	 * <p>The editor input.</p>
	 */
	private ITraceEditorInput m_editorInput;
	/**
	 * <p>The widget factory to use in the creation of this editor's contents.</p>
	 */
	private TraceWidgetFactory m_widgetFactory;
	
	// The main widget area of this editor.
	private Form m_mainForm;

	static {
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				if (!PlatformUI.isWorkbenchRunning()) {
					SAUILSPlugin.getDefault().logInfo("Failed to add automatic editor refresh.  Workbench not running."); //$NON-NLS-1$
					return;
				}
				//Register the SA part listener for refresh actions.
				PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().addPartListener(
						new RefreshListener());
			}
		});
	}
	
	@Override
	public void dispose() {
		GCUtil.gc();
		synchronized(this) {
			if (m_widgetFactory != null) {
				m_widgetFactory.dispose();
			}
		}
		super.dispose();
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
	}
	
	/**
	 * 
	 * @param id
	 * @param helpContextId help context id for the editor, may be <b>null</b>
	 */
	public AbstractSaEditor(String id, String helpContextId) {
		m_id = id;
		m_helpContextId = helpContextId;
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
	}
	
	/**
	 * 
	 * Gives access to the part name from outside the part
	 * @param name
	 */
	public void setEditorTitle(String title) {
		setPartName(title);
	}
		
	@Override
	public void createPartControl(Composite parent) {
		
		// create the main form for the editor client area
		TraceWidgetFactory factory = null;
		synchronized(this) {
			factory = getWidgetFactory();
		}

		m_mainForm = factory.createForm(parent);
		factory.decorateFormHeading(m_mainForm);
		
		// set the layout; the body area of the form is the one 
		// that should be filled with the child widgets
		Composite childArea = m_mainForm.getBody();
		childArea.setLayout(new FillLayout());
		
		m_mainForm.setImage(getHeaderImage());
		
		m_mainForm.setText(getHeaderText());
		
		// Create the specific controls for the editor.
		createPartControl0(childArea);
		
		factory.paintBordersFor(childArea);
		m_mainForm.layout();
	
		// Set the help.  Note that PlatformUI.getWorkbench() throws during
		// our TraceTimelineTest, so we check getSite() to avoid the throw.
		if (getSite() != null) {
			IWorkbench workbench = PlatformUI.getWorkbench();
			if (m_helpContextId != null && workbench != null) {
				IWorkbenchHelpSystem helpSystem = workbench.getHelpSystem();
				helpSystem.setHelp(parent, m_helpContextId);
			}
		} else {
			SAUILSPlugin.getDefault().logInfo("Editor " + getId() + " has no site."); //$NON-NLS-1$ //$NON-NLS-2$
		}
		
		// Mark creation of SWT controls as finished.
		m_createSwtControlsReady.countDown();
	}
	
	/**
	 * Create the specific controls for the SA Editor.
	 * 
	 * TODO make this abstract after MCU LogEditor is updated
	 * @param parent
	 * @return
	 */
	abstract protected void createPartControl0(Composite parent);
	
	/**
	 * Determine if the editor needs to refresh the input data.
	 * 
	 * @return - true if new data should be loaded.
	 */
	public boolean needsRefresh() {
		return m_needsRefresh;
	}

	/**
	 * Set editor state.
	 * 
	 * @param needsRefresh - used to determine if this editor needs to be reloaded.
	 */
	public void shouldRefresh(boolean needsRefresh) {
		m_needsRefresh = needsRefresh;
	}
	
	/**
	 * To be implemented in all SA editors.
	 */
	public abstract void refresh();
	
	/**
	 * @return the unique id of this SA editor
	 */
	public String getId() {
		return m_id;
	}
	
	/*
	 * Returns a base name used to identify the group this view belongs to.
	 * At this time the base name is the launch config base name.
	 * */
	public abstract String getBaseName();	
	

	/*
	 * Returns the path to the folder location of the input file.
	 * This information it is used to close/refresh editors
	 * 
	 * */
	public abstract String getStoragePath();

	@Override
	protected void setInput(IEditorInput input) {
		super.setInput(input);
		if (input instanceof ITraceEditorInput) {
			m_editorInput = (ITraceEditorInput) input;
		}
		else if (!(input instanceof FileStoreEditorInput)
				&& !(input instanceof FileEditorInput)
				&& !(input instanceof IPathEditorInput)) {
			SAUILSPlugin.getDefault().logInfo("Unexpected trace editor input type " + input.getClass().getSimpleName()); //$NON-NLS-1$
		}
	}

	@Override
	public void resourceChanged(IResourceChangeEvent event) {
		
		switch (event.getType()) {
		case IResourceChangeEvent.PRE_CLOSE:
			if (getProjectName().equalsIgnoreCase(event.getResource().getName())) {
				closeFile();
			}
			break;

		case IResourceChangeEvent.PRE_DELETE:
			if (getProjectName().equalsIgnoreCase(event.getResource().getName())) {
				closeFile();
			}
			break;
		case IResourceChangeEvent.POST_CHANGE:
		
			IResourceDeltaVisitor visitor = new IResourceDeltaVisitor() {
				public boolean visit(IResourceDelta delta) {
					IResource resource = delta.getResource();

					switch (delta.getKind()) {
						case IResourceDelta.REMOVED:
							switch (resource.getType()) {
	
								case IResource.FOLDER: {
									if (resource.getProject().getName().equalsIgnoreCase(getProjectName())
											&& resource.getName().equalsIgnoreCase(TRACE_FOLDER)) {
										closeFile();
									}
									break;
								}
		
								case IResource.FILE: {
									/*
									 * If the file associated with this view is deleted,
									 * close the view with the exception when the view is opened 
									 * with tarmac or dat names, e.g. deleting the tarmac should not 
									 * cause the view to close
									 */
									IPath resourcePath = resource.getLocation();
									IPath editorFilePath = new Path(getStoragePath());
									editorFilePath = editorFilePath.append(getPartName());
									if (resourcePath != null && resourcePath.getFileExtension() == null 
											|| editorFilePath.getFileExtension() == null) {
										break;
									}
									
									if (isDecodableToCsv(resourcePath.getFileExtension())) {
										break;
									}

									if (resourcePath.equals(editorFilePath)) {
										closeFile();
									}
									
									// if the csv is deleted, close the corresponding dat or tarmac file editor
									if (resourcePath.getFileExtension().equals(trace_csv_ext.substring(1))) {
										String resourceFileName = resourcePath.toOSString();
										String editorFileName = editorFilePath.toOSString();
										
										String resourceFileNameWithoutExtension = resourceFileName.substring(0, resourceFileName.lastIndexOf('.'));
										String editorFileNameWithoutExtension = editorFileName.substring(0, editorFileName.lastIndexOf('.'));
										if (isDecodableToCsv(editorFilePath.getFileExtension())
												&& resourceFileNameWithoutExtension.equals(editorFileNameWithoutExtension)) {
											closeFile();
										}
									}
									
									break;
								}
								default: break;
							}
						break;
					}
					return true; // visit the children
				}
			};

			try {
				event.getDelta().accept(visitor);
			} catch (CoreException e) {
				SAUILSPlugin.getDefault().logError("Error while treating resource changed event.", e);
			}
		default: break;
		}
	}
	
	private void closeFile() {
		Display.getDefault().syncExec(new Runnable() {
			
			public void run() {
				IWorkbenchPartSite site = AbstractSaEditor.this.getSite();
				if (site == null) {
					return;
				}
				site.getPage().closeEditor(AbstractSaEditor.this, false);
							
			}
		});
	}		
		
	public String getHeaderText() {
		
		// the below should be safe as the type of the editor input is checked at init time
		if (m_editorInput != null && getEditorType() != null) {
			String sampleName = m_editorInput.getName();
			Path dirPath = new Path(m_editorInput.getStorageDirectory());
			String contextName = dirPath.removeLastSegments(1).lastSegment();
			if (contextName != null)
				return getEditorType() + " - " + sampleName + " - " + contextName; //$NON-NLS-1$ //$NON-NLS-2$
			else
				return getEditorType() + " - " + sampleName; //$NON-NLS-1$
		}

		return AbstractSaEditor_DefaultText;
	}
	
	public abstract String getEditorType();
	public abstract Image getHeaderImage();
	
	protected Form getMainForm() {
		return m_mainForm;
	}
	
	public synchronized TraceWidgetFactory getWidgetFactory() {
		if (m_widgetFactory == null) {
			m_widgetFactory = new TraceWidgetFactory();
		}
			
		return m_widgetFactory;
	}
	
	protected void setProject(FileEditorInput fileInput) {
		IProject proj = fileInput.getFile().getProject();
		m_project = proj;
	}
	
	/**
	 * Set am associated project. With this a poject can be set even if the opened file is outside the project
	 * 
	 * @param proj - the project
	 */
	public void setProject(IProject proj) {
		m_project = proj;
	}
	
	/*
	 * Returns a project name used to identify the group this view belongs to.
	 * */
	public String getProjectName() {
		if (m_project != null) {
			return m_project.getName();
		}
		
		return CommonConstants.EMPTY_STRING;
	}
	
	
	/**
	 * Used to identify files that must be decoded to a 
	 * csv file prior to being displayed
	 * @param fileExtension file extension without dot
	 * @return true if the file type must be decoded to a csv prior to opening the view 
	 */
	private boolean isDecodableToCsv(String fileExtension) {
		
		return fileExtension.equals(trace_data_ext.substring(1)) 
				|| fileExtension.equals(trace_tarmac_ext.substring(1))
				|| fileExtension.equals(trace_cwzsa_ext.substring(1))
				|| fileExtension.equals(trace_kcwzsa_ext.substring(1))
				|| fileExtension.equals(trace_scwzsa_ext.substring(1));
							
	}
}
